## ************************************************************************************************
## aggregateRadialFiles function definition
## ----------------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics
##  - used for aggregating cell counting by tree, regrouping the radial files using median or mean.
##
## Arguments:
##   - data: a data.frame containing cell count data. This data frame must contain 12 imposed variables
##      with compulsory column names: Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, PR.
##   - stat: a optional character string indicating if RF file should be aggregated
##      using "median" (default) or "mean".
##
## Outputs:
##   - A data frame composed of 13 columns containing
##      the unchanged variables: Site, Year, Species, Tree, Sample, DY
##		  as well as the computed means for CZ, EZ, WZ, MZ, PR, and sums for WMZ and EWMZ.
##
## Versions:
##    1.0.1. Compute mean and cumulative number of cells.
##		2.0.1. Implementing group treatment by tree and year.
##    3.0.1. Up-date using the new imposed names for variables.
##   	3.1-1. Changing the names of the input variables.
##   	3.1-2. New update (2016/09/27)
##
## Started: 25 March 2010
## Last modifications: 29 September 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

aggregateRadialFiles <- function(data, stat = "median") {
  
   # message("--> Entering aggregateRadialFiles function...")
   
   ## Declaration and settings
   ## ------------------------
   DF <- data
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Species <- as.factor(DF$Species)
	DF$Tree <- as.factor(DF$Tree)
   DF$Sample <- as.factor(DF$Sample)
	
   ## Creating a new aggregated dataframe by radial files
   ## ---------------------------------------------------
	## Case 1: option stat is "mean"
	if (stat == "mean") {
      ODF <- aggregate(DF[, c("CZ", "EZ", "WZ", "MZ", "PR")],
                       by=list(Site=DF$Site, Year=DF$Year, Species=DF$Species, Tree=DF$Tree, Sample=DF$Sample, DY=DF$DY),
                       FUN=mean, na.rm=TRUE)
	} # End if (Stat == "mean")

   ## Case 2: option stat is "median" (default)
   if (stat == "median") {
      ODF <- aggregate(DF[, c("CZ", "EZ", "WZ", "MZ", "PR")],
                       by=list(Site=DF$Site, Year=DF$Year, Species=DF$Species, Tree=DF$Tree, Sample=DF$Sample, DY=DF$DY),
                       FUN=median, na.rm=TRUE)
      } # End if (Stat == "median")

	## Ordering the new aggregated data frame
	## --------------------------------------
	ODF <- ODF[order(ODF$Site, ODF$Year, ODF$Tree, ODF$Sample), ]
	
	## Computing cumulative cell number
	## --------------------------------
	## WMZ: number of cells in wall-thickening phenophase + mature cells
	## EWMZ: number of cells in expansion + wall-thickening + mature phenophases
	ODF$WMZ <- ODF$WZ + ODF$MZ
	ODF$EWMZ <- ODF$EZ + ODF$WZ + ODF$MZ

	## Rounding values
	## ---------------
	ODF[, c("CZ", "EZ", "WZ", "MZ", "PR", "WMZ", "EWMZ")] <-
      round(ODF[, c("CZ", "EZ", "WZ", "MZ", "PR", "WMZ", "EWMZ")], digits=1)	
	
	## Returning output dataframe
	## --------------------------
	return(ODF)
	
} ## End aggregateRadialFiles function 

## ------------------------------------------------------------------------------------------------
##                           End aggregateRadialFiles function
## ************************************************************************************************
