##*********************************************************************************************
## computeGompertzCellTimings() function definition
## -------------------------------------------------
##
## Purpose: Data analysis function for computing individual cell timings from a set of Gompertz parameters
##
## Arguments:
## 		gompar: a dataframe containing three sets of Gompertz parameters (a, b, k)
##
## Output:
##		a dataframe containing the timings of the cells
##
## Versions:
## 		1. Basic cell timings computation
##
## Current Version: 1.1-0
## Started: 11 June 2012
## Last modifications: 19 June 2012
##
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------


computeGompertzCellTimings <- function(gompar) {

	## Reversing Gompertz function to cumpute dates from cell numbers
	revGompertz <- function(y, a, b, k) {t <- (b - log(log((a+1)/y))) / k}

	## Beginning of enlarging phase (nELM curve)
	## -----------------------------------------
	a1 <- gompar$a[1]
	b1 <- gompar$b[1]
	k1 <- gompar$k[1]
	N1 <- 1:a1
	tE <- revGompertz(N1, a1, b1, k1)

	## Beginning thickening phase (nLM curve)
	## --------------------------------------
	a2 <- gompar$a[2]
	b2 <- gompar$b[2]
	k2 <- gompar$k[2]
	N2 <- 1:a2
	tL <- revGompertz(N2, a2, b2, k2)

	## Beginning Mature phase (nM curve)
	## ---------------------------------
	a3 <- gompar$a[3]
	b3 <- gompar$b[3]
	k3 <- gompar$k[3]
	N3 <- 1:a3
	tM <- revGompertz(N3, a3, b3, k3)

	## Durations
	## ---------
	dE <- tL - tE
	dL <- tM - tL
	dT <- tM - tE
	
	## Constructing the resulting data frame
	N <- min(N1, N2, N3)
	CT <- data.frame(N, tE, tL, tM, dE, dL, dT)
	return(CT)
	
} ## End computeGompertzCellTimming() function
##*********************************************************************************************