## ************************************************************************************************
## computeRingCellNb function definition
## ------------------------------------------------------------------------------------------------
##
## Purpose -> Data processing function for computing final total ring cell number.
## 
## Arguments:
##	   - data: a data frame composed of 13 columns containing the descriptive variables (Site, Year,
##            Species, Tree, Sample, and DY; as well as the quantitative variables (CZ, EZ, WZ, MZ).
##            Typically, this data frame is the output of the aggregateRadialFiles or standardiseCellCounts
##            functions, alternativelly it can also work on raw data.
##		- stat: an optional character indicating if "mean" or "median" (default) shoudl be computed.
##		
##
## Outputs:
##		- A data frame composed of ? columns containing the group variables Tree and Year
##		  as well as the computed values of RCN, ICN & FCN : median with median absolute deviation,
##		  median absolute error and normalized median absolute error
##		  or mean with standard deviation and standard error;
##		  the number of observations and the signal to noise ration are also provided.
##
##
## Versions:
##		1.0-0 Compute RCN, ICN, FCN
##		   1.1. Bug on RCN computation corrected
##		   1.2. Adding se and mae
##		   1.3. Modifying the function so it could work on raw or averaged cell count table
##		   1.4. Adding signal-to-noise ratio (SNR)
##			1.4.1. Allowing merge to produce NA's in the output table
##		2.1-0. Implementing group treatment by tree and year
##    3.1-0. General cleanning
##       3.1-1. Changing variable names (2016-09-29)
##       3.1-2. Testing computation of signal to noise ratio to ICN and FCN -- Concelled (2016-09-30)
##       3.1-3. MAD is Mean Absolute Difference and not Median Absolute Difference anymore,
##               This is to avoid creating Inf during SNR computation (2016-10-06)
##
## Started: 25 March 2010
## Last modifications: 06 October 2016
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

computeRingCellNb <- function(data, stat = "median") {
   
   #message("--> Entering computeRingCellNb function...")
   
   ## =============================================================================================   
   ## Declarations and settings
   ## =============================================================================================
   ## Acronyms
   ## --------
   ## Median: Median
   ## Median Absolute Deviation: MAD
   ## Median Absolute Error: MAE
   ## Median Absolute Error with normal correction: MAE.cor
   ## Mean: Mean
   ## standard deviation: SD
   ## standard error: SE
   ## signal-to-noise ration: SNR
   
   
	## Ensuring that all the group variables are factors
	## -------------------------------------------------
   DF  <- data
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Tree <- as.factor(DF$Tree)
   DF$Species <- as.factor(DF$Species)
	
	## New data frame declaration
   ## --------------------------
	RCN <- data.frame()
	ICN <- data.frame()
	FCN <- data.frame()
	
	## Selecting the data needed for the calculation of RCN, ICN and FCN
	## =================================================================
	## RCN (final total Ring Cell Number)
	## ----------------------------------
	RCN1  <- DF[is.na(DF$EZ) == FALSE & DF$EZ < 0.1
	            & is.na(DF$MZ) == FALSE & DF$MZ >= 1,
	            c("Site", "Year", "Tree", "Species", "DY", "WZ", "MZ")]
	RCN1$TCN <- RCN1$WZ + RCN1$MZ
	RCN1$One <- 1
	
	## ICN (Initial cambial Cell Number)
	## ---------------------------------
	ICN1  <- DF[is.na(DF$CZ) == FALSE
	            & is.na(DF$EZ) == FALSE & DF$EZ < 0.1
	            & is.na(DF$MZ) == FALSE & DF$MZ < 0.1,
	            c("Site", "Year", "Tree", "Species", "DY", "CZ")]
	ICN1$One <- 1
	
	## FCN (Fnitial cambial Cell Number)
	## ---------------------------------
	FCN1  <- DF[is.na(DF$CZ) == FALSE
	            & is.na(DF$EZ) == FALSE & DF$EZ < 0.1
	            & is.na(DF$MZ) == FALSE & DF$MZ >= 1,
	            c("Site", "Year", "Tree", "Species", "DY", "CZ")]
	FCN1$One <- 1	
	
	## ============================================================================================= 
	## Option "median" (by default)
	## ============================================================================================= 
	if (stat=="median") {
	   
	   ## Computing RCN median and median absolute deviation, plus the signal-to-noise ratio
	   ## ==================================================================================
	   ## Aggregating data frames
	   ## -----------------------
      RCN2 <- aggregate(RCN1[, "TCN"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
                        median, na.rm=TRUE)
	   names(RCN2) <- c("Site", "Year", "Tree", "Species", "RCN.Median")
   
      RCN3 <- merge(RCN1, RCN2, by=c("Site", "Year", "Tree", "Species"))
      RCN3$AbsDif <- abs(RCN3$TCN - RCN3$RCN.Median)
      
      RCN4 <- aggregate(RCN3[, "AbsDif"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
                        mean, na.rm=TRUE)
      names(RCN4) <- c("Site", "Year", "Tree", "Species", "RCN.MAD")
   
      RCN5 <- aggregate(RCN1[, "One"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
                        sum, na.rm=TRUE)
      names(RCN5) <- c("Site", "Year", "Tree", "Species", "RCN.CNT")
   
      ## Merging the resulting data frames
      ## ---------------------------------
      RCNa <- merge(RCN2, RCN4, by=c("Site", "Year", "Tree", "Species"))
      RCN  <- merge(RCNa, RCN5, by=c("Site", "Year", "Tree", "Species"))
   
      ## Computing the median absolute error and the signal-to-noise ratio
      ## -----------------------------------------------------------------
      RCN$RCN.MAE <- RCN$RCN.MAD / sqrt(RCN$RCN.CNT)
      RCN$RCN.MAE.cor <- 3/4 * RCN$RCN.MAE
      RCN$RCN.SNR <- RCN$RCN.Median / RCN$RCN.MAD
   
      ## Rounding the results
      ## --------------------
      RCN$RCN.MAE <- round(RCN$RCN.MAE, digits=2)
      RCN$RCN.MAE.cor <- round(RCN$RCN.MAE.cor, digits=2)
      RCN$RCN.SNR <- round(RCN$RCN.SNR, digits=2)
      
      ## Computing ICN median and median absolute deviation
      ## ==================================================
      ## Aggregating data frames
      ## -----------------------
      ICN2 <- aggregate(ICN1[, "CZ"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
                        median, na.rm=TRUE)
      names(ICN2) <- c("Site", "Year", "Tree", "Species", "ICN.Median")
   
      ICN3 <- merge(ICN1, ICN2, by=c("Site", "Year", "Tree", "Species"))
      ICN3$AbsDif <- abs(ICN3$CZ - ICN3$ICN.Median)
      
      ICN4 <- aggregate(ICN3[, "AbsDif"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
                        mean, na.rm=TRUE)
      names(ICN4) <- c("Site", "Year", "Tree", "Species", "ICN.MAD")
   
      ICN5 <- aggregate(ICN1[, "One"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
                        sum, na.rm=TRUE)
      names(ICN5) <- c("Site", "Year", "Tree", "Species", "ICN.CNT")
   
      ## Merging the resulting data frames
      ## ---------------------------------
      ICNa <- merge(ICN2, ICN4, by=c("Site", "Year", "Tree", "Species"))
      ICN  <- merge(ICNa, ICN5, by=c("Site", "Year", "Tree", "Species"))
   
      ## Computing the median absolute error and and the signal-to-noise ratio
      ## ---------------------------------------------------------------------
      ICN$ICN.MAE <- ICN$ICN.MAD / sqrt(ICN$ICN.CNT)
      ICN$ICN.MAE.cor <- 3/4 * ICN$ICN.MAE
   
      ## Rounding the results
      ## --------------------
      ICN$ICN.MAE <- round(ICN$ICN.MAE, digits=2)
      ICN$ICN.MAE.cor <- round(ICN$ICN.MAE.cor, digits=2)
   
      ## Computing FCN median and median absolute deviation
      ## ==================================================
      ## Aggregating data frames
      ## -----------------------
      FCN2 <- aggregate(FCN1[, "CZ"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
                        median, na.rm=TRUE)
      names(FCN2) <- c("Site", "Year", "Tree", "Species", "FCN.Median")
   
      FCN3 <- merge(FCN1, FCN2, by=c("Site", "Year", "Tree", "Species"))
      FCN3$AbsDif <- abs(FCN3$CZ - FCN3$FCN.Median)
      
      FCN4 <- aggregate(FCN3[, "AbsDif"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
                        mean, na.rm=TRUE)
      names(FCN4) <- c("Site", "Year", "Tree", "Species", "FCN.MAD")
   
      FCN5 <- aggregate(FCN1[, "One"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
                        sum, na.rm=TRUE)
      names(FCN5) <- c("Site", "Year", "Tree", "Species", "FCN.CNT")
   
      ## Merging the resulting data frames
      ## ---------------------------------
      FCNa <- merge(FCN2, FCN4, by=c("Site", "Year", "Tree", "Species"))
      FCN  <- merge(FCNa, FCN5, by=c("Site", "Year", "Tree", "Species"))
   
      ## Computing the median absolute error and the signal-to-noise ratio
      ## -----------------------------------------------------------------
      FCN$FCN.MAE <- FCN$FCN.MAD / sqrt(FCN$FCN.CNT)
      FCN$FCN.MAE.cor <- 3/4 * FCN$FCN.MAE
   
      ## Rounding the results
      ## --------------------
      FCN$FCN.MAE <- round(FCN$FCN.MAE, digits=2)
      FCN$FCN.MAE.cor <- round(FCN$FCN.MAE.cor, digits=2)
      
   
	## ============================================================================================= 
	## Option "mean"
	## ============================================================================================= 
	} else if (stat=="mean") {
		
		## Computing RCN mean and standard error, plus the signal-to-noise ratio
		## =====================================================================
	   ## Aggregating data frames
	   ## -----------------------
		RCN2 <- aggregate(RCN1[, "TCN"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
		                  mean, na.rm=TRUE)
		names(RCN2) <- c("Site", "Year", "Tree", "Species", "RCN.Mean")
	
		RCN3 <- aggregate(RCN1[, "TCN"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
		                  sd, na.rm=TRUE)
		names(RCN3) <- c("Site", "Year", "Tree", "Species", "RCN.SD")
		
		RCN4 <- aggregate(RCN1[, "One"], by=list(Site=RCN1$Site, Year=RCN1$Year, Tree=RCN1$Tree, Species=RCN1$Species),
		                  sum, na.rm=TRUE)
		names(RCN4) <- c("Site", "Year", "Tree", "Species", "RCN.CNT")
		
		## Merging the resulting data frames
		## ---------------------------------
		RCNa <- merge(RCN2, RCN3, by=c("Site", "Year", "Tree", "Species"))
		RCN  <- merge(RCNa, RCN4, by=c("Site", "Year", "Tree", "Species"))
		
		## Computing the standard error and the signal-to-noise ratio
		## ----------------------------------------------------------
		RCN$RCN.SE <- RCN$RCN.SD /sqrt(RCN$RCN.CNT)
		RCN$RCN.SNR <- RCN$RCN.Mean / RCN$RCN.SD
		
		## Rounding the results
		## --------------------
		RCN$RCN.Mean <- round(RCN$RCN.Mean, digits=2)
		RCN$RCN.SD <- round(RCN$RCN.SD, digits=2)
		RCN$RCN.SE <- round(RCN$RCN.SE, digits=2)
		RCN$RCN.SNR <- round(RCN$RCN.SNR, digits=2)
	
		## Computing ICN mean and standard error
		## =====================================
		## Aggregating data frames
		## -----------------------
		ICN2 <- aggregate(ICN1[, "CZ"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
		                  mean, na.rm=TRUE)
		names(ICN2) <- c("Site", "Year", "Tree", "Species", "ICN.Mean")
	
		ICN3 <- aggregate(ICN1[, "CZ"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
		                  sd, na.rm=TRUE)
		names(ICN3) <- c("Site", "Year", "Tree", "Species", "ICN.SD")
		
		ICN4 <- aggregate(ICN1[, "One"], by=list(Site=ICN1$Site, Year=ICN1$Year, Tree=ICN1$Tree, Species=ICN1$Species),
		                  sum, na.rm=TRUE)
		names(ICN4) <- c("Site", "Year", "Tree", "Species", "ICN.CNT")
		
		## Merging the resulting data frames
		## ---------------------------------
		ICNa <- merge(ICN2, ICN3, by=c("Site", "Year", "Tree", "Species"))
		ICN  <- merge(ICNa, ICN4, by=c("Site", "Year", "Tree", "Species"))
		
		## Computing the standard error and the signal-to-noise ratio
		## ----------------------------------------------------------
		ICN$ICN.SE <- ICN$ICN.SD /sqrt(ICN$ICN.CNT)
		#ICN$ICN.SNR <- ICN$ICN.Mean / ICN$ICN.SD
		
		## Rounding the results
		## --------------------
		ICN$ICN.Mean <- round(ICN$ICN.Mean, digits=2)
		ICN$ICN.SD <- round(ICN$ICN.SD, digits=2)
		ICN$ICN.SE <- round(ICN$ICN.SE, digits=2)
		#ICN$ICN.SNR <- round(ICN$ICN.SNR, digits=2)
		
		## Computing the mean FCN and its standard error
		## =============================================
		## Aggregating data frames
		## -----------------------
		FCN2 <- aggregate(FCN1[, "CZ"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
		                  mean, na.rm=TRUE)
		names(FCN2) <- c("Site", "Year", "Tree", "Species", "FCN.Mean")
	
		FCN3 <- aggregate(FCN1[, "CZ"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
		                  sd, na.rm=TRUE)
		names(FCN3) <- c("Site", "Year", "Tree", "Species", "FCN.SD")
	
		FCN4 <- aggregate(FCN1[, "One"], by=list(Site=FCN1$Site, Year=FCN1$Year, Tree=FCN1$Tree, Species=FCN1$Species),
		                  sum, na.rm=TRUE)
		names(FCN4) <- c("Site", "Year", "Tree", "Species", "FCN.CNT")
		
		## Merging the resulting data frames
		## ---------------------------------
		FCNa <- merge(FCN2, FCN3, by=c("Site", "Year", "Tree", "Species"))
		FCN  <- merge(FCNa, FCN4, by=c("Site", "Year", "Tree", "Species"))
		
		## Computing the standard error and the signal-to-noise ratio
		## ----------------------------------------------------------
		FCN$FCN.SE <- FCN$FCN.SD /sqrt(FCN$FCN.CNT)
		#FCN$FCN.SNR <- FCN$FCN.Mean / FCN$FCN.SD
		
		## Rounding the results
		## --------------------
		FCN$FCN.Mean <- round(FCN$FCN.Mean, digits=2)
		FCN$FCN.SD   <- round(FCN$FCN.SD, digits=2)
		FCN$FCN.SE   <- round(FCN$FCN.SE, digits=2)
		#FCN$FCN.SNR <- round(FCN$FCN.SNR, digits=2)
		
	## ============================================================================================= 
	## Problem signaling
	## ============================================================================================= 
	} else {
	   stop(paste('no option available for stat = "', stat, '", please check the spelling!'))
	} ## end else
	
	## ============================================================================================= 
	## Preparing the results
	## ============================================================================================= 	
	## Merging data frames
	## -------------------
	ODF1 <- merge(RCN, ICN, by=c("Site", "Year", "Tree", "Species"), all=TRUE)
	ODF2 <- merge(ODF1, FCN, by=c("Site", "Year", "Tree", "Species"), all=TRUE)
	
	## Ordering output data frame
	ODF <- ODF2[order(ODF2$Site, ODF2$Year, ODF2$Tree), ]
	
	## Returning the results
	return(ODF)

} ## End function computeRingCellNb

## ------------------------------------------------------------------------------------------------
##                           End computeRingCellNb function
## ************************************************************************************************