## *********************************************************************************************************
## drawSmartGompertzCurve.IUF function definition
## ----------------------------------------------
##
## Purpose: Visualisation function for plotting a Gompertz curves along with its enveloppe,
##           for a group of trees on an existing plot.
##
## Arguments:
##   - gompertz.simulations: a data frames containing the model simulations for all the day of the year.
##		 This is typically the output of fitGompertzModels function.
##   - colors: a numerical vector contining the colors code (three values).
##
## Output:
##   - One Gompertz curve showing the mean of the group along with its standard error enveloppe.
##      the function also represents the values of the timings
##
## Versions:
##		1.1-0. Defining this new function (29/11/2016)
##		1.1-1. Checking this new function (01/12/2016)
##    1.1-2 Improvement and homogeneisation (12/04/2017)
##
## Started: 29 November 2016
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

drawSmartGompertzCurve.IUF <- function(gompertz.simulations, colors=NULL) {

   #message("--> Entering drawSmartGompertzCurve.IUF function...")

   ## Data manipulation and plot settings
   ## ***********************************
   ## Reading input data
   ## ------------------
   IDF <- gompertz.simulations
   
   ## Defining color code
   ## -------------------
   if (is.null(colors)) {
      Col.env <- hsv(h=0.6, s=0.3, v=0.9, alpha=0.75) ## Color for the enveloppe
      Col.cur <- hsv(h=0.6, s=0.6, v=0.8, alpha=1) ## Color for the mean curve
      Col.pts <- hsv(h=0.6, s=0.9, v=0.7, alpha=1) ## Color for the points and timing bars
   } else {
      Col.env <- colors[1] ## Color for the enveloppe
      Col.cur <- colors[2] ## Color for the mean curve
      Col.pts <- colors[3] ## Color for the points and timing bars
   }

   ## Computing the mean and the mean standard error
   ## ==============================================
   ## Defining a function to compute the standard error of the mean
   ## -------------------------------------------------------------
   sem <- function(x) {
      sem <- sd(x, na.rm=TRUE) / sqrt(length(x))
   }
   
   ## Computing the mean
   ## ------------------
   MNDF <- aggregate(IDF$EWMZ, by=list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species, DY=IDF$DY), mean)
   names(MNDF) <- c("Site", "Year", "Species", "DY", "Mean")
   
   ## Computing the sem
   ## ------------------
   SEDF <- aggregate(IDF$EWMZ, by=list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species, DY=IDF$DY), sem)
   names(SEDF) <- c("Site", "Year", "Species", "DY", "SEM")
   
   ## Creating a new data frame with the mean and the sem
   ## ---------------------------------------------------
   GEDF <- merge(MNDF, SEDF)
   GEDF <- GEDF[order(GEDF$DY), ]
   
   ## Computing high and low bound for the enveloppe
   ## ----------------------------------------------
   GEDF$HB <- GEDF$Mean + GEDF$SEM
   GEDF$LB <- GEDF$Mean - GEDF$SEM
   
   ## Fitting new Gompertz functions on the mean and high and low bounds
   ## ====================================================================
   ## Fitting a Gompertz function for the mean
   ## ----------------------------------------
   R.fitGPZ.mean <- fitGPZ(GEDF$DY, GEDF$Mean)
   
   ## Recording the timings for the mean Gompertz curve
   am <- R.fitGPZ.mean$coef["a"]
   tim <- R.fitGPZ.mean$coef["ti"]
   tfm <- R.fitGPZ.mean$coef["tf"]
   txm <- R.fitGPZ.mean$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.mean <- R.fitGPZ.mean$sim
   
   ## Fitting a Gompertz function for the high bound
   ## ----------------------------------------------
   R.fitGPZ.hb <- fitGPZ(GEDF$DY, GEDF$HB)
   
   ## Recording the timings for the high Gompertz curve
   ahb <- R.fitGPZ.hb$coef["a"]
   tihb <- R.fitGPZ.hb$coef["ti"]
   tfhb <- R.fitGPZ.hb$coef["tf"]
   txhb <- R.fitGPZ.hb$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.hb <- R.fitGPZ.hb$sim
   
   ## Fitting a Gompertz function for the low bound
   ## ---------------------------------------------
   R.fitGPZ.lb <- fitGPZ(GEDF$DY, GEDF$LB)
   
   ## Recording the timings for the low Gompertz curve
   alb <- R.fitGPZ.lb$coef["a"]
   tilb <- R.fitGPZ.lb$coef["ti"]
   tflb <- R.fitGPZ.lb$coef["tf"]
   txlb <- R.fitGPZ.lb$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.lb <- R.fitGPZ.lb$sim
   
   ## Creating a new dataframe containing mean and high and low bounds
   ## ----------------------------------------------------------------
   SIMDF <- data.frame(DY=SIMDF.mean$DY, Mean=SIMDF.mean$Sim, HB=SIMDF.hb$Sim, LB=SIMDF.lb$Sim)
   
   ## Plotting the data
   ## *****************
   ## Drawing the enveloppe of the mean Gompertz curve
   ## ------------------------------------------------
   ## Creating a new data frame for plotting the polygon
   PLGdf <- data.frame(X=c(SIMDF$DY, SIMDF$DY[rev(order(SIMDF$DY))]),
                       Y=c(SIMDF$LB, SIMDF$HB[rev(order(SIMDF$HB))]))
   
   ## Drawing the enveloppe
   polygon(PLGdf$X, PLGdf$Y, col=Col.env, border=NA)
   
   ## Plotting the Gompertz curve for the mean of the group
   ## =====================================================   
   ## Plotting the curve
   lines(SIMDF$DY, SIMDF$Mean, col=Col.cur, lwd=2, lty=1)
   
   ## Plotting additional informations
   ## ================================
   ## Plotting the timings on the curve
   ## ---------------------------------
   points(tim, 0.05*am, type="p", pch=17, col=Col.pts, cex=1.5)
   points(txm, am/exp(1), type="p", pch=16, col=Col.pts, cex=1.5)
   points(tfm, 0.95*am, type="p", pch=15, col=Col.pts, cex=1.5)
   
   ## Drawing the timmings on the x-axis
   ## ----------------------------------
   rug(seq(from=min(tihb, tilb), to=max(tihb, tilb), by=0.25), col=Col.env, lwd=1)
   rug(tim, col=Col.pts, lwd=2)
   rug(seq(from=min(txhb, txlb), to=max(txhb, txlb), by=0.25), col=Col.env, lwd=1)
   rug(txm, col=Col.pts, lwd=2)
   rug(seq(from=min(tfhb, tflb), to=max(tfhb, tflb), by=0.25), col=Col.env, lwd=1)
   rug(tfm, col=Col.pts, lwd=2)
   
   ## Writing labels for the timing signs
   ## -----------------------------------
   mtext(expression(alpha), side=4, line=0.25, at=am, col=Col.pts, las=1)
   mtext(expression(t[b]), side=1, line=0.25, at=tim, col=Col.pts)
   mtext(expression(t[x]), side=1, line=0.25, at=txm, col=Col.pts)
   mtext(expression(t[c]), side=1, line=0.25, at=tfm, col=Col.pts)

} ## End function drawSmartGompertzCurve.IUF

## ------------------------------------------------------------------------------------------------
##                           End drawSmartGompertzCurve.IUF function
## ************************************************************************************************