## ************************************************************************************************
## expandCCT.RF function definition
## --------------------------------
##
## Purpose: Internal function of expandCellCountTable()
##  - used for expanding cell count tables containing RF
##       so that all the missing values are coded explicitly with NA
##
## Arguments:
##   - data: a data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, PR.
##
## Outputs:
##   - a data.frame with the same structure as the argument.
##
## Versions:
##   1.0-0. Defining this new function
##   1.0-1. Changing the names of the input variables
##
## Started: 1 July 2014
## Last modifications: 30 June 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

expandCCT.RF <- function(data) {

   # message("--> Entering expandCCT.RF function...")
   DF <- data

   ## Expanding input data in order to make all missing values explicit with NA
   ## =========================================================================
   ## Breaking the implicit transmission of factor levels
   ## --> Replace Site.u by fSite?
   fSite <- factor(as.character(DF$Site))
   fYear <- factor(as.character(DF$Year))
   fTree <- factor(as.character(DF$Tree))
   fSpecies <- factor(as.character(DF$Species))
   fSample <- factor(as.character(DF$Sample))
   fDY <- factor(as.character(DF$DY))
   fRF <- factor(as.character(DF$RF))

   ## Expanding Sample factor for each potential Radial File level and vice versa
   Sample.for.RF <- rep(levels(fSample), each=nlevels(fRF))
   RF.for.Sample <- rep(levels(fRF), times=nlevels(fSample))
   
   ## Expanding Tree factor for each potential combination of Radial File and Sample factors
   Tree.for.Sample.and.RF <- rep(levels(fTree), each=length(RF.for.Sample))
   Sample.for.Tree.and.RF <- rep(Sample.for.RF, times=nlevels(fTree))
   RF.for.Tree.and.Sample <- rep(RF.for.Sample, times=nlevels(fTree))
   
   # Expanding Year factor for each potential combination of Tree, Sample and RF factors
   Year.for.Tree.Sample.and.RF <- rep(levels(fYear), each=length(RF.for.Tree.and.Sample))
   Tree.for.Year.Sample.and.RF <- rep(Tree.for.Sample.and.RF, times=nlevels(fYear))
   Sample.for.Year.Tree.and.RF <- rep(Sample.for.Tree.and.RF, times=nlevels(fYear))
   RF.for.Year.Tree.and.Sample <- rep(RF.for.Tree.and.Sample, times=nlevels(fYear))
   
   # Expanding Site factor for each potential combination of Year, Tree, Sample and RF factors
   Site.for.Year.Tree.Sample.and.RF <- rep(levels(fSite), each=length(RF.for.Year.Tree.and.Sample))
   Year.for.Site.Tree.Sample.and.RF <- rep(Year.for.Tree.Sample.and.RF, times=nlevels(fSite))
   Tree.for.Site.Year.Sample.and.RF <- rep(Tree.for.Year.Sample.and.RF, times=nlevels(fSite))
   Sample.for.Site.Year.Tree.and.RF <- rep(Sample.for.Year.Tree.and.RF, times=nlevels(fSite))
   RF.for.Site.Year.Tree.and.Sample <- rep(RF.for.Year.Tree.and.Sample, times=nlevels(fSite))

   ## Constructing a data frame for the expanded factors
   EDF1 <- data.frame(Site=Site.for.Year.Tree.Sample.and.RF, Year=Year.for.Site.Tree.Sample.and.RF,
                      Tree=Tree.for.Site.Year.Sample.and.RF, Sample=Sample.for.Site.Year.Tree.and.RF,
                      RF=RF.for.Site.Year.Tree.and.Sample)
   
   ## Adding Species to the expanded data frame
   SpDF  <- unique(DF[, c("Species","Tree")])
   EDF2 <- merge(EDF1, SpDF, all=TRUE)
   
   ## Adding  DY to the expanded data frame
   DYDF  <- unique(DF[, c("Sample","DY")])
   EDF3 <- merge(EDF2, DYDF, all=TRUE)
   
   # Ordering the expanded data frame
   EDF4 <- data.frame(Site=EDF3$Site,  Year=EDF3$Year, Tree=EDF3$Tree, Species=EDF3$Species,
                     Sample=EDF3$Sample, DY=EDF3$DY, RF=EDF3$RF)
   
   # Constructing the final expanded data frame containing cell count data
   CDF <- DF[, c("Site", "Year", "Tree", "Sample", "RF", "CZ", "EZ", "WZ", "MZ", "PR")]
   EDF5 <- merge(EDF4, CDF, by=c("Site", "Year", "Tree", "Sample", "RF"), all=TRUE)
   ODF <- EDF5[order(EDF5$Site, EDF5$Year, EDF5$Tree, EDF5$DY, EDF5$RF), ]
   
   return(ODF)
   
} ## End expandCCT.RF function

## ------------------------------------------------------------------------------------------------
##                           End expandCCT.RF function
## ************************************************************************************************
