## ************************************************************************************************
## expandCellCountTable function definition
## ----------------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics()
##  - used for expanding cell count tables so that all the missing values are coded explicitly with NA
##
## Arguments:
##   - data: a data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, CZ, EZ, WZ, MZ, PR
##
## Outputs:
##   - a data.frame with the same structure as the argument
##
## Versions:
##   1.0-0. Defining this new function
##   1.0-1. Changing the names of the input variables
##
## Started: 1 July 2014
## Last modifications: 30 June 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

expandCCT.Tree <- function(data) {

   # message("--> Entering expandCCT.Tree function...")
   DF <- data
   
   ## Expanding input data in order to make all missing values explicit with NA
   ## =========================================================================
   ## Breaking the implicit transmission of factor levels
   ## --> Replace Site.u by fSite?
   fSite <- factor(as.character(DF$Site))
   fYear <- factor(as.character(DF$Year))
   fTree <- factor(as.character(DF$Tree))
   fSpecies <- factor(as.character(DF$Species))
   fSample <- factor(as.character(DF$Sample))
   fDY <- factor(as.character(DF$DY))
   
   ## Expanding Tree factor for each potential Sample levels and vice versa
   Tree.for.Sample <- rep(levels(fTree), each=nlevels(fSample))
   Sample.for.Tree <- rep(levels(fSample), times=nlevels(fTree))
   
   # Expanding Year factor for each potential combination of Tree and Sample factors
   Year.for.Sample.and.Tree <- rep(levels(fYear), each=length(Sample.for.Tree))
   Tree.for.Sample.and.Year <- rep(Tree.for.Sample, times=nlevels(fYear))
   Sample.for.Tree.and.Year <- rep(Sample.for.Tree, times=nlevels(fYear))
   
   # Expanding Site factor for each potential combination of Year, Tree and Sample factors
   Site.for.Sample.and.Tree.and.Year <- rep(levels(fSite), each=length(Sample.for.Tree.and.Year))
   Year.for.Sample.and.Tree.and.Site <- rep(Year.for.Sample.and.Tree, times=nlevels(fSite))
   Tree.for.Sample.and.Year.and.Site <- rep(Tree.for.Sample.and.Year, times=nlevels(fSite))
   Sample.for.Tree.and.Year.and.Site <- rep(Sample.for.Tree.and.Year, times=nlevels(fSite))
   
   ## Constructing a data frame for the expanded factors
   FDF1 <- data.frame(Site=Site.for.Sample.and.Tree.and.Year, Year=Year.for.Sample.and.Tree.and.Site,
                      Tree=Tree.for.Sample.and.Year.and.Site, Sample=Sample.for.Tree.and.Year.and.Site)
   
   ## Adding Species to the expanded data frame
   UDF1  <- unique(DF[, c("Species","Tree")])
   FDF2 <- merge(FDF1, UDF1, all=TRUE)
   
   ## Adding  DY to the expanded data frame
   UDF2  <- unique(DF[, c("Sample","DY")])
   FDF3 <- merge(FDF2, UDF2, all=TRUE)
   
   # Ordering the data frame of expanded factors
   FDF <- data.frame(Site=FDF3$Site,  Year=FDF3$Year, Tree=FDF3$Tree, Species=FDF3$Species,
                     Sample=FDF3$Sample, DY=FDF3$DY)
   FDF <- FDF[order(FDF$Site, FDF$Year, FDF$Tree, FDF$DY), ]
   
   # Constructing the final expanded data frame containing cell count data
   CDF <- DF[, c("Site", "Year", "Tree", "Sample", "CZ", "EZ", "WZ", "MZ", "PR")]
   EDF <- merge(FDF, CDF, by=c("Site", "Year", "Tree", "Sample"), all=TRUE)
   EDF <- EDF[order(EDF$Site, EDF$Year, EDF$Tree, EDF$DY), ]
   
   return(EDF)
   
} ## End function expandCCT.Tree

## ------------------------------------------------------------------------------------------------
##                           End expandCCT.Tree function
## ************************************************************************************************
