## *********************************************************************************************************
## fitGPZ function definition
## ---------------------------
##
## Purpose
##   Internal function of fitGomp used for fitting a Gompertz model to one individual cumulative cell count.
##
## Arguments
##   - D: a vector containing the dates in day of year.
##   - N: a vector containing the number of cells.
##   - A: an optional numeric containing the value of the asymptote.
##   - start: an optional named numerical vector containing the seeds for a, b, and k parameters.
##   - plot.title: an optional character containing the plot title
##   - plot.fitting: an optional bolean indicating if a graph of the fitting must be plotted or not (default).
##   - plot.obs.vs.pre: an optional bolean indicating if a verification graph must be plotted or not (default).
##
## Outputs
##   - a list containing a numeric vector of the computed coeficients and the values of obs and pred.
##
## Versions
##   1.1-1. Fit Gompertz on cumulated number of Cells and plot a graphic of the fittings
##          and a verification plot if asked
##   2.2-1. Implementing starting values option
##   2.2-2. Changing the labels of the axes
##   2.2-3. General cleaning (27-31/10/2016)
##   2.2-4. Updating Gompertz parameter names (04/11/2016)
##   2.2-5. Debugging output list construction (10/11/2016)
##   2.2-6. Changing function name (11/11/2016)
##   2.2-7. Modifying output variable names (21/11/2016)
##   2.3-1. Adding a new table with simulations in the output list (23/11/2016)
##   2.3-2. Bug correction (24/11/2016)
##	  2.3-3. New Checking (02/12/2016)
##	  2.3-4. Changing variables names (12/04/2017)
##   2.3-5. Improvement and homogeneisation (12/04/2017)
##
## Started: 27 November 2008
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

fitGPZ <- function(D, N, A = NULL, start = NULL, plot.title = NULL,
                   plot.fitting = FALSE, plot.obs.vs.pre = FALSE) {
   
   # message("--> Entering fitGPZ function...")

	## Defining local variables and checking incoming data
	## ===================================================
   ## axis labels
	lblx <- "Day of Year"
	lbly <- "Cell number"
	
	## Checking data consistency
   if(length(D) != length(N)) stop("dates and number of Cells are not of the same lenght!")
	
	
	## Gompertz fitting
	## ****************
	## Initialising starting values for Gompertz fitting
	if (is.null(start) == TRUE) { start=c(a=max(N, na.rm=TRUE), b=7, k=0.04) }
	
	## Gompertz fitting when Gompertz asymptote is initiated by the maximal number of cells
	## ------------------------------------------------------------------------------------
	if (is.null(A) == TRUE) {
	   ## b and k parameters initiated using value from Rossi et al. 2003
	   Gompertz <- nls(N ~ a*exp(-exp(b - k*D)), start=start,
	                   control=list(maxiter = 100, warnOnly = TRUE))
	   
	   ## if maxiter is reached sending the reference of the trees in addition to the previous warning
	   if (Gompertz$convInfo$finIter >= 100) { warning(plot.title) }
	   
	   ## Extracting the fitted parameters
	   a <- as.double(coef(Gompertz)["a"])
	   b <- as.double(coef(Gompertz)["b"])
	   k <- as.double(coef(Gompertz)["k"])
	} ## End if (is.null(A) == TRUE)

	## Gompertz fitting when Gompertz asymptote is fixed by the user
	## -------------------------------------------------------------
	if (is.null(A) == FALSE) {
		## b and k parameters initiated using value from Rossi et al. 2003
		Gompertz <- nls(N ~ A*exp(-exp(b - k*D)), start=start[2:3],
		                control=list(maxiter = 100, warnOnly = TRUE))
		
		## if maxiter is reached sending the reference of the trees in addition to the previous warning
		if (Gompertz$convInfo$finIter >= 100) { warning(plot.title) }
		
		## Extracting the fitted parameters
		a <- A
		b <- as.double(coef(Gompertz)["b"])
		k <- as.double(coef(Gompertz)["k"])
	} ## End if (is.null(A) == FALSE)
	
	## Computing the biological parameters from the fitted equations
	## -------------------------------------------------------------
	## ti: date at which 5% of the cells are produced
	ti <- round((b - log(-log(0.05)))/k, digits=0)
	
	## tx: date of inflexion point
	tx <- round(b/k, digits=0)
	
	## tf: date at which 95% of the cells are produced
	tf <- round((b - log(-log(0.95)))/k, digits=0)
	
	## dt: time elapsed between tf and ti (mean growth duration)
	dt <- round(log(log(0.05)/log(0.95))/k, digits=0)
	
	## rx: maximal growth rate (cell/day)
	rx <- round((k*a)/exp(1), digits=2)
	
	## ra: mean growth rate computed between 5 and 95% of the produced cells (cell/day)
	ra <- round((0.9*k*a)/(log(log(0.05)/log(0.95))), digits=2)
	
	## Computing simulated values
	## --------------------------	
	## Computing the predicted values for comparison with obeserved data
	G <- predict(Gompertz, list(D=D))
	
	## Computing daily predicted values
	DS <- 1:365
	S <- predict(Gompertz, list(D=DS))
	
	## Evaluating the goodness-of-fit
	## ------------------------------	
	## Constructing a data.frame to remove missing values
	GDF <- data.frame(D, N, G)
	GDF <- na.omit(GDF)
	
	## Computing the R2
	R2 <- round(cor(GDF$N, GDF$G)^2, digits=2)
		
	## Computing modelling efficiency
	EF <- round(1 - sum((GDF$N - GDF$G)^2) / sum((GDF$N - mean(GDF$N))^2), digits=2)
		
	## Computing root mean squared deviation
	RMSD <- round(sqrt(1/(length(GDF$N) - 1) * sum((GDF$N - GDF$G)^2)), digits=2)	
	
		
	## Recording the results of the fitting
	## ------------------------------------
	## Preparing gompertz parameters
	a <- round(a, digits=2)
	b <- round(b, digits=2)
	k <- round(k, digits=4)

	## compiling the results
	## ---------------------
	R <- c(a, b, k, ti, tf, dt, tx, ra, rx, R2, EF, RMSD)
	names(R) <- c("a", "b", "k", "ti", "tf", "dt", "tx", "ra", "rx", "R2", "EF", "RMSD")
	
	
	## Ploting the data and the fitted Gompertz curve if option plot.fitting=TRUE
	## **************************************************************************
	if (plot.fitting==TRUE) {
		
		## Setting the plot region
		ymax <- max(c(N, (G + G/5)), na.rm=TRUE)
		xmin <- min(D)
		xmax <- max(D)
		plot(D, N, type="n", xlim=c(xmin, xmax), ylim=c(0, ymax), ann=FALSE, axes=FALSE)

		## Drawing individual curves
		points(D, N, type="p", pch=19, col="blue")
		GF <- function(x) a*exp(-exp(b - k*x))
		curve(GF, from=xmin, to=xmax, add=TRUE, col="red")
	
		## Customising axes
		axis(1)
		mtext(lblx, side=1, line=2.5)
	
		axis(2)
		mtext(lbly, side=2, line=2.5)

		## Writting plot title
		mtext(plot.title, side=3, line=2, adj=0.0, cex=1.0, font=2)
		mtext(paste("R2 = ", round(R2, digits=2)), side=3, line=0.5, adj=0.0, cex=1.0)
	
		## Additional labels
		## -----------------
		adjx <- 2 * (xmax - xmin)/100

		## Asymptote
		abline(h=a, lty=2, col="darkgreen")
		mtext(expression(alpha), side=4, line=0.25, at=a, col="darkgreen", las=1)

		## tb (date for 5% of growth)
		points(c(ti, ti), c(-10, 0.05*a), type="l", lty=3, col="darkgreen")
		mtext(expression(t[b]), side=1, line=-1.5, at=(ti+adjx) , col="darkgreen")
	
		## tc (date for 95% of growth)
		points(c(tf, tf), c(-10, 0.95*a), type="l", lty=3, col="darkgreen")
		mtext(expression(t[c]), side=1, line=-1.5, at=(tf+adjx), col="darkgreen")

		## tx ((date of maximal growth)
		VLXi <- c(tx, tx)
		VLYi <- c(-10, a/exp(1))
		points(VLXi, VLYi, type="l", lty=2, col="darkgreen")
		mtext(expression(t[x]), side=1, line=-1.5, at=(tx+adjx), col="darkgreen")

		## Mean speed
		arrows(ti, 0.05*a, tf, 0.95*a, length=0.15, angle=25, code=2, col="darkgreen")
		xt <- 2 * adjx + ((ti + tf) / 2)
		text(xt, 0.5*a, expression(g[m]), col="darkgreen")

		## Maximal speed
		alpha <- atan(rx)
		xs <- (rx) * cos(alpha)
		ys <- sqrt(rx^2 - xs^2)
		c <- 50
		arrows(tx, a/exp(1), (tx + c*xs), (a/exp(1) + c*ys), length=0.15,
			angle=25, code=2, col="darkgreen")
		text(tx - adjx, 0.5*a, expression(g[x]), col="darkgreen")
	
		## Legend
		xmin <- min(D)
		r <- ymax / 15
		points(xmin, ymax, pch=19, col="blue")
		text(xmin + 1, ymax, "Observations", pos=4)
		points(xmin, ymax - r, pch= "-", col="red")
		text(xmin + 1, ymax - r, "Gompertz model", pos=4)
	
		box()
		
		## Adding CaviaR stamp on the figure
		mtext("CaviaR.2 - fitGompertz function",
		      side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
		
	} ## End if (plot.fitting==TRUE)
		
		
	## Ploting the observed vs. predicted values if option plot.obs.vs.pre = TRUE
	## **************************************************************************
	if (plot.obs.vs.pre==TRUE) {
	   
	   ## Setting the plot region
	   xymax <- max(c(N, G), na.rm=TRUE)
					
		## Plotting the data
		plot(G, N, type="p", xlim=c(0, xymax), ylim=c(0, xymax), pch=19, col="blue", ann=FALSE, axes=FALSE)
			
		## Plotting the xy line and the regression line
		abline(a=0, b=1, lty=5, col="darkgreen")
		reg1 <- lm(N ~ G)
		if (is.na(reg1$coef[1]) == FALSE & is.na(reg1$coef[2]) == FALSE) { abline(reg1, lty=2, col="red") }
	
		## Customising axes
		axis(1)
		mtext("Predictions (Pdt)", side=1, line=2.5)
	
		axis(2)
		mtext("Observations (Obs)", side=2, line=2.5)

		## Writting plot title
		mtext(plot.title, side=3, line=2, adj=0.0, cex=1.0, font=2)
		mtext(paste("R2 = ", round(R2, digits=2)), side=3, line=0.5, adj=0.0, cex=1.0)
				
		## Verification tests
		mtext(paste("Obs = ", round(reg1$coef[2], digits=2), " Pdt + ",
			round(reg1$coef[1], digits=0), sep=""), side=3, line=-2, at=0, adj=0, cex=1.0)
			
		diff <- G - N
		reg2 <- lm(diff ~ G)
		res2 <- summary(reg2)
		t1 <- NA
		if (is.na(reg2$coef[1]) == FALSE) { t1 <- round(res2$coef[1, 4], digits=2) }
		t2 <- NA
		if (is.na(reg2$coef[2]) == FALSE) { t2 <- round(res2$coef[2, 4], digits=2) }
			
		mtext(paste("Significance of test a = 0: p.value = ", t1, sep=""), side=3, line=-3,
			at=0, adj=0, cex=1.0)
		mtext(paste("Significance of test b = 1: p.value = ", t2, sep=""), side=3, line=-4,
			at=0, adj=0, cex=1.0)
	
		box()
		
		## Adding CaviaR stamp on the figure
		mtext("CaviaR.2 - fitGompertz function",
		      side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
		
	} ## End if (plot.obs.vs.pre==TRUE)
	
	## Returning results
	## -----------------
	return(list(coef=R, val=data.frame(DY=D, Obs=N, Pred=G), sim=data.frame(DY=DS, Sim=S)))

}  ## End fitGPZ

## ------------------------------------------------------------------------------------------------
##                           End fitGPZ function
## ************************************************************************************************