## *********************************************************************************************************
## fitGompertzModels function definition
## -------------------------------------
##
## Purpose
##   Data analysis function for fitting a Gompertz model to cumulative cell count data.
##
## Arguments
##   - data: a dataframe containing the grouping variables (Tree and Year),
##			as well as the dates in day of year (DY),
##			along with at least one (but potentially three) cumulative cell count number (EWMZ, WMZ, and MZ).
##   - parameters: an optional data.frame containing the grouping variables,
##			and the value of the asymptotes (A - the same value will be used for EWMZ, WMZ, and MZ),
##			and/or the values of the Gompertz parameter seeds (a, b, k).
##   - predictions: an optional bolean indicating if predicted values must be returned or not (default).
##   - plot.fitting: an optional bolean indicating if a graph of the fitting must be plotted or not (default).
##   - plot.obs.vs.pre: an optional bolean indicating if a verification graph must be plotted or not (default).
##
## Output
##   - by default: a dataframe containing the fitting coeficients only,
##      if option predictions = TRUE: a list of three data frames containing:
##         1. the references of the group (ref);
##         2. the fitting coefficients (coef); and
##			  3. the values of the observations and predictions (values).
##
## Versions:
##   1.1-1. fit EWMZ for data count tables.
##	  2.1-1. fit EWMZ for individual tree. 
##	  3.1-1. fit EWMZ, WMZ, and MZ for individual tree.
##	  4.1-1. Takes starting values for the fitting as optional argument.
##	  5.1-1. Use Tree and Year as compulsory grouping variables.
##	  5.2-1. General cleaning (27-31/10/2016).
##	  5.2-2. Updating Gompertz parameter names (04/11/2016).
##   5.2-3. debugging output list construction (10/11/2016).
##   5.2-4. Changing function name (11/11/2016).
##   5.2-5. Modifying output variable names (21/11/2016).
##   5.3-1. Adding a new table with simulations in the output list (23/11/2016).
##	  5.3-2. New Checking (02/12/2016).
##
## Started: 27 November 2008
## Last modifications: 23 November 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

fitGompertzModels <- function(data, parameters = NULL, predictions = FALSE,
                              plot.fitting = FALSE, plot.obs.vs.pre = FALSE) {
   
   #message("--> Entering fitGompertzModels function...")
   
   ## Formating incoming data
   ## -----------------------
	CDF <- data
	PDF <- parameters
	fita <- FALSE
	
	
	## Processing the data
	## ===================
	## Creating a new data frame with Gompertz parameters if provided
	## --------------------------------------------------------------
	if (is.null(PDF) == FALSE) {
		TDF <- merge(CDF, PDF, by=c("Site", "Year", "Tree", "Species"), all=FALSE)		
		fita <- TRUE
	} else {TDF <- CDF}
	
	## Calling fitGompertz function to fit a Gompertz model to each of the three cumulative cell counts
	## ------------------------------------------------------------------------------------------------
	RLST <- by(TDF, TDF[c("Site", "Year", "Species", "Tree")],
	           fitGompertz, fit.A=fita, plot.fitting=plot.fitting, plot.obs.vs.pre=plot.obs.vs.pre)

	## Cleaning the result list from NULL elements
	OLST <- RLST[ ! sapply(RLST, is.null) ]
	
	
	## Returning the results
	## =====================
	## Formating the results for option prediction == FALSE
	## ----------------------------------------------------
	if (predictions == FALSE) {
	   
	   ## Extracting the coefficients from OLST and collecting them in a data frame
	   ODF <- getGompertzCoefficients.IUF(OLST)
	   
	   ## Returning the output data frame
	   return (ODF)
	   
	## Formating the results for option prediction == TRUE
	## ----------------------------------------------------		
	} else { return(OLST) }

} ## End fitGompertzModels

## ------------------------------------------------------------------------------------------------
##                           End fitGompertzModels function
## ************************************************************************************************