## *********************************************************************************************************
## fitGompertz function definition
## -------------------------------
##
## Purpose
##   internal function of fitGompertz used for fitting Gompertz models on cumulative cell count number: EWMZ, WMZ, MZ.
##
## Arguments
##   - data: a data frame containing the dates in day of year (DY) along with
##			at least one (but potentially three) cumulative cell count number: EWMZ, WMZ, MZ.
##			Optionnally data may also contain the asymptote value, if fit.A=TRUE,
##			and/or the seeds (a, b, k) for the Gompertz fitting.
##   - fit.A: an optional bolean indicating if asymptote values are provided in the input data,
##			   or must bit fitted (default).
##   - start: an optional named numeric vector containing the seeds (a, b, k) for the Gompertz fitting.
##   - plot.fitting: an optional bolean indicating if a graph of the fitting must be plotted or not (default).
##   - plot.obs.vs.pre: an optional bolean indicating if a verification graph must be plotted or not (default).
##
## Output
##   - a list of three data.frames:
##      - the first one (references) containing references;
##      - the second one (coefficients) containing the fitted parameters;
##      - the third one (values) containing the observed and predicted values.
##
## Versions
##   1.1-1. fit EWMZ, WMZ, and MZ for tree/year.
##   2.2-1. General cleaning and changing variable names (27-31/10/2016).
##   2.2-2. Updating Gompertz parameter names (04/11/2016).
##   2.2-3. Debugging output list construction (10/11/2016).
##   2.2-4. Changing function name (11/11/2016).
##   2.2-5. Modifying output variable names (21/11/2016).
##   2.3-1. Adding a new table with simulations in the output list (23/11/2016).
##	  2.3-2. New Checking (02/12/2016).
##
## Started: 27 November 2008
## Last modifications: 23 November 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

fitGompertz <- function(data, fit.A = FALSE, plot.fitting = FALSE, plot.obs.vs.pre = FALSE) {
   
   # message("--> Entering fitGompertz function...")
   
   ## Preparing cell count data
   ## =========================
   ## Formating incoming data
   ## -----------------------
	DF <- data
	
	## Initiating start argument
	if (is.null(list(DF$a, DF$b, DF$k)) == FALSE) {start=c(a=DF$a[1] , b=DF$b[1], k=DF$k[1])}
	else { start=NULL }
	
	## Initiating main argument which will be used to construct plot.title
	main <- paste("Tree", unique(DF$Tree), " - Year ", unique(DF$Year))
		
	## Initiating object to be returned
	CoF <- data.frame(matrix(data=NA, nrow=3, ncol=13))
	names(CoF) <- c("Type", "a", "b", "k", "ti", "tf", "dt", "tx", "ra", "rx", "R2", "EF", "RMSD")
		
	VoF <- data.frame(matrix(data=NA, nrow=nrow(DF), ncol=7))
	names(VoF) <- c("DY", "EWMZ.obs", "EWMZ.pred",
					"WMZ.obs", "WMZ.pred", "MZ.obs", "MZ.pred")
	
	SoF <- data.frame(matrix(data=NA, nrow=365, ncol=4))
	names(SoF) <- c("DY", "EWMZ.sim", "WMZ.sim", "MZ.sim")
	
	
	## Fitting Gompertz models when A is not provided
	## ==============================================	
	if (fit.A == FALSE) {

		## fitting Gompertz for EWMZ
	   ## -------------------------
		if(is.null(list(DF$DY, DF$EWMZ)) == FALSE) {
			type <- "EWMZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$EWMZ, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[1, 1] <- type
			CoF[1, 2:13] <- R.fitGPZ$coef
			VoF[, 1:3] <- R.fitGPZ$val
			SoF[, 1:2] <- R.fitGPZ$sim
		}

		## fitting Gompertz for WMZ
	   ## ------------------------
		if(is.null(list(DF$DY, DF$WMZ)) == FALSE) {
			type <- "WMZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$WMZ, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[2, 1] <- type
			CoF[2, 2:13] <- R.fitGPZ$coef
			VoF[, 4:5] <- R.fitGPZ$val[, 2:3]
			SoF[, 3] <- R.fitGPZ$sim[, 2]
		}

		## fitting Gompertz for MZ
	   ## -----------------------
		if(is.null(list(DF$DY, DF$MZ)) == FALSE) {
			type <- "MZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$MZ, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[3, 1] <- type
			CoF[3, 2:13] <- R.fitGPZ$coef
			VoF[, 6:7] <- R.fitGPZ$val[, 2:3]
			SoF[, 4] <- R.fitGPZ$sim[, 2]
		}
	} ## End if (fit.A==FALSE)

	
	## Fitting Gompertz models when A is provided
	## ==========================================
	if (fit.A == TRUE) {
		
		A <- DF$A[1]

		## fitting Gompertz for EWMZ
		## -------------------------
		if(is.null(list(DF$DY, DF$EWMZ)) == FALSE) {
			type <- "EWMZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$EWMZ, A=A, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[1, 1] <- type
			CoF[1, 2:13] <- R.fitGPZ$coef
			VoF[, 1:3] <- R.fitGPZ$val
			SoF[, 1:2] <- R.fitGPZ$sim
		}

		## fitting Gompertz for WMZ
		## ------------------------
		if(is.null(list(DF$DY, DF$WMZ)) == FALSE) {
			type <- "WMZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$WMZ, A=A, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[2, 1] <- type
			CoF[2, 2:13] <- R.fitGPZ$coef
			VoF[, 4:5] <- R.fitGPZ$val[, 2:3]
			SoF[, 3] <- R.fitGPZ$sim[, 2]
		}

		## fitting Gompertz for MZ
		## -----------------------
		if(is.null(list(DF$DY, DF$MZ)) == FALSE) {
			type <- "MZ"
			R.fitGPZ <- fitGPZ(DF$DY, DF$MZ, A=A, start=start,
							plot.fitting=plot.fitting, plot.title=paste(main, " - ", type),
							plot.obs.vs.pre=plot.obs.vs.pre)
			CoF[3, 1] <- type
			CoF[3, 2:13] <- R.fitGPZ$coef
			VoF[, 6:7] <- R.fitGPZ$val[, 2:3]
			SoF[, 4] <- R.fitGPZ$sim[, 2]
		}
	} ## End if (fit.A==TRUE)
	
	## Removing empty rows
	#CoF <- na.omit(CoF)

	## Warning in case no fitting was done
	if (nrow(CoF) == 0) warning("No data available, no Gompertz fitted!")
	
	## Returning the results
	## ---------------------
	References <- c(Site=as.character(unique(DF$Site)),
	                Year=as.integer(as.character(unique(DF$Year))),
	                Species=as.character(unique(DF$Species)),
	                Tree=as.integer(as.character(unique(DF$Tree))))
	                

	return(list(references=References, coefficients=CoF, values=VoF, simulations=SoF))

}  ## End fitGompertz

## ------------------------------------------------------------------------------------------------
##                           End fitGompertz function
## ************************************************************************************************