## ************************************************************************************************
## FUNCTION overviewCellCountTable()
## ---------------------------------
##
## FUNCTION DEVELOPMENT
## --------------------
##
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
## Purpose: Summarizing the content of a cell count dataframe
## Version:
##  1.1. First version
##	   1.1-0. Homogeneization of variable names
##	   1.1-1. Changing variable names again and improving output
##
## Started: 22 January 2016
## Last modifications: 27 June 2016
##
##
## FUNCTION DEFINITION
## -------------------
##
## Basic function to summarize cell count dataframe content
## 
## Arguments:
##   - data: data.frame with imposed column names:
##       Site, Year, Tree, Species, Sample, DY, RF, CZ , EZ, WZ, MZ, PR.
##
## Output:
##		- summary of cell count table.
##
## ================================================================================================
overviewCellCountTable <- function(data) {
   
   ## reading input data
   ## ------------------
   IDF <- data

   ## ************************************************************************************************ 
   ## 1. Checking the presence of compulsory variables and converting them accordingly to future use
   ## ************************************************************************************************
   ## --> Checking that all the variables needed are present.
   ##     --> Sending a warning if it is not the case.
   ##     --> Converting them correctly if it is the case.
   
   ## Initialisation of objects needed further
   ## ----------------------------------------
   IDF.col.names <- names(IDF)
   all.comp.var.present <- TRUE
   
   ## Compulsory variable Site
   ## ------------------------
   if ("Site" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'Site' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$Site <- as.factor(IDF$Site)
   
   ## Compulsory variable Year
   ## ------------------------
   if ("Year" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'Year' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$Year <- as.factor(IDF$Year)
   
   ## Compulsory variable Tree
   ## ------------------------
   if ("Tree" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'Tree' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$Tree <- as.factor(IDF$Tree)
   
   ## Compulsory variable Species
   ## ---------------------------
   if ("Species" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'Species' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$Species <- as.factor(IDF$Species)
   
   ## Compulsory variable Sample
   ## --------------------------
   if ("Sample" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'Sample' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$Sample <- as.factor(IDF$Sample)
   
   ## Compulsory variable DY
   ## ----------------------
   if ("DY" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'DY' not found!")
      all.comp.var.present <-  FALSE
   }
   
   ## Compulsory variable RF
   ## ----------------------
   if ("RF" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'RF' not found!")
      all.comp.var.present <-  FALSE
   } else IDF$RF <- as.factor(IDF$RF)
   
   ## Compulsory variable Ca
   ## ----------------------
   if ("CZ" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'CZ' not found!")
      all.comp.var.present <-  FALSE
   }
   
   ## Compulsory variable EZ
   ## ----------------------
   if ("EZ" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'EZ' not found!")
      all.comp.var.present <-  FALSE
   }
   
   ## Compulsory variable WZ
   ## ----------------------
   if ("WZ" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'WZ' not found!")
      all.comp.var.present <-  FALSE
   }
   
   ## Compulsory variable MZ
   ## ----------------------
   if ("MZ" %in% IDF.col.names == FALSE) {
      message("!!! Compulsory variable 'MZ' not found!")
      all.comp.var.present <-  FALSE
   }
   
   ## Optional variable PR
   ## --------------------
   if ("PR" %in% IDF.col.names == FALSE) {
      message("!!! Optional variable 'PR' not found!")
   }
   
   ## Writting a final message if all the required variables are present
   ## ------------------------------------------------------------------
   if (all.comp.var.present == TRUE) {
      message("--> All the compulsory variables are present in the dataframe.")
   } else stop("--> All the compulsory variables are not present in the dataframe.")
   
   
   ## ************************************************************************************************ 
   ## 2. Checking the format of variables
   ## ************************************************************************************************
   ## --> Checking that all the variables needed can be converted in the good format
   ##        or sending a warning if it is not the case...
   
   ## Initialisation of objects needed further
   ## ----------------------------------------
   all.var.cor.type <- TRUE
   
   ## Compulsory variable Site
   ## ------------------------
   if (is.factor(IDF$Site) == FALSE) {
      message("! Compulsory variable 'Site' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable Year
   ## ------------------------
   if (is.factor(IDF$Year) == FALSE) {
      message("! Compulsory variable 'Year' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable Tree
   ## ------------------------
   if (is.factor(IDF$Tree) == FALSE) {
      message("! Compulsory variable 'Tree' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable Species
   ## ---------------------------
   if (is.factor(IDF$Species) == FALSE) {
      message("! Compulsory variable 'Species' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable Sample
   ## --------------------------
   if (is.factor(IDF$Sample) == FALSE) {
      message("! Compulsory variable 'Sample' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable DY
   ## ----------------------
   if (is.integer(IDF$DY) == FALSE) {
      message("! Compulsory variable 'DY' should be an integer!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable RF
   ## ----------------------
   if (is.factor(IDF$RF) == FALSE) {
      message("! Compulsory variable 'RF' should be a factor!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable CZ
   ## ----------------------
   if (is.numeric(IDF$CZ) == FALSE) {
      message("! Compulsory variable 'CZ' should be an integer!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable EZ
   ## ----------------------
   if (is.numeric(IDF$EZ) == FALSE) {
      message("! Compulsory variable 'EZ' should be an integer!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable WZ
   ## ----------------------
   if (is.numeric(IDF$WZ) == FALSE) {
      message("! Compulsory variable 'WZ' should be an integer!")
      all.var.cor.type <- FALSE
   }
   
   ## Compulsory variable MZ
   ## ----------------------
   if (is.numeric(IDF$MZ) == FALSE) {
      message("! Compulsory variable 'MZ' should be an integer!")
      all.var.cor.type <- FALSE
   }
   
   ## Optional variable PR
   ## --------------------
   if (is.numeric(IDF$PR) == FALSE) {
      message("! Optional variable 'PR' should be a numeric!")
      all.var.cor.type <- FALSE
   }
   
   ## Writting a final message if all the variables are of the correct type
   ## ---------------------------------------------------------------------
   if (all.var.cor.type == TRUE) {
      message("--> All the variables of the dataframe are in the correct format.")
   } else message("--> Some variables of the dataframe can not be converted in the correct format")
   
   
   ## ************************************************************************************************ 
   ## 3. Overview of the factors and quantitative variables
   ## ************************************************************************************************
   ## --> providing a short summary of the factors and quantitative variables of the data frame
   
   ## Factors
   ## =======
   ## Site
   ## ----
   number.of.sites <- paste("Number of sites:", nlevels(IDF$Site), sep=" ")
   name.of.sites <- paste("Site names:", paste(levels(IDF$Site), collapse=", "), sep=" ")
   message("")
   message(number.of.sites)
   message(name.of.sites)
   
   ## Year
   ## ----
   number.of.years <- paste("Number of years:", nlevels(IDF$Year), sep=" ")
   name.of.years <- paste("Years of monitoring:", paste(levels(IDF$Year), collapse=", "), sep=" ")
   message("")
   message(number.of.years)
   message(name.of.years)
   
   ## Tree
   ## ----
   number.of.trees <- paste("Number of trees:", nlevels(IDF$Tree), sep=" ")
   name.of.trees <- paste("Tree names:", paste(levels(IDF$Tree), collapse=", "), sep=" ")
   message("")
   message(number.of.trees)
   message(name.of.trees)
   
   ## Species
   ## -------
   number.of.species <- paste("Number of species:", nlevels(IDF$Species), sep=" ")
   name.of.species <- paste("Species names:", paste(levels(IDF$Species), collapse=", "), sep=" ")
   message("")
   message(number.of.species)
   message(name.of.species)
   
   ## Sample and DY
   ## -------------
   number.of.samples <- paste("Number of sampling day per year:", nlevels(IDF$Sample), sep=" ")
   name.of.samples <- paste("Indices for sampling days:", paste(levels(IDF$Sample), collapse=", "), sep=" ")
   name.of.days <- paste("Corresponding day of year:", paste(levels(as.factor(IDF$DY)), collapse=", "), sep=" ")
   message("")
   message(number.of.samples)
   message(name.of.samples)
   message(name.of.days)
   
   ## RF
   ## --
   number.of.radial.files <- paste("Number of counted radial files:", nlevels(IDF$RF), sep=" ")
   name.of.radial.files <- paste("Indices for radial files:", paste(levels(IDF$RF), collapse=", "), sep=" ")
   message("")
   message(number.of.radial.files)
   message(name.of.radial.files)
   
   ## Quantitative variables
   ## ======================
   ## Computing the final number of observations
   ## ------------------------------------------
   NbObs.CZ <- length(na.omit(IDF$CZ))
   NbObs.EZ <- length(na.omit(IDF$EZ))
   NbObs.WZ <- length(na.omit(IDF$WZ))
   NbObs.MZ <- length(na.omit(IDF$MZ))
   if ("PR" %in% IDF.col.names == TRUE) NbObs.PR <- length(na.omit(IDF$PR))
   
   ## Reporting the summary and the final number of observations
   ## ----------------------------------------------------------
   message("")
   message("Summary for quantitative variables")
   message("----------------------------------")
   if ("PR" %in% IDF.col.names == TRUE) {
      print(summary(IDF[c("CZ", "EZ", "WZ", "MZ", "PR")]))
      message(paste(" Nb. Obs.: ", NbObs.CZ, " Nb. Obs.: ", NbObs.EZ, " Nb. Obs.: ", NbObs.WZ,
                    " Nb. Obs.: ", NbObs.MZ, " Nb. Obs.: ", NbObs.PR))
   } else {
      print(summary(IDF[c("CZ", "EZ", "WZ", "MZ")]))
      message(paste(" Nb. Obs.: ", NbObs.CZ, " Nb. Obs.: ", NbObs.EZ, " Nb. Obs.: ", NbObs.WZ,
                    " Nb. Obs.: ", NbObs.MZ))      
   }
   
} ## end of overviewCellCountTable function definition
## ************************************************************************************************
   