## ************************************************************************************************
## plotCDC() function definition
## --------------------------------------------
##
## Internal function for plotting Cleaveland Dot Chart for a set of cell counts data
## 
## Arguments:
##    - x: a vector containing cell count data
##    - plot.type: a character containing the type of cells for the label for the plot and other uses
##		- hue: a numeric containing the value of hue (between 0 and 1)
##
## Outputs:
##		- A single plot of Cleaveland Dot Chart for cell counts data
##    - A vector containing the values for the lower and upper fences
##
## Versions:
##		1.0.0 Function developemnt started.
##    1.0-1. 2014-10-09 --- Fixing typos.
##    1.1.0. 2015-12-15 --- Changing plot title.
##    1.1.1. 2016-06-22 --- Changing variable names.
##    1.1.2. 2016-06-24 --- Handling case where there is no data to plot.
##    1.1.3. 2016-06-24 --- Including zero and negative values in outliners.
##
## Started: 22 Jully 2014
## Last modifications:  24 June 2016
## Author: Cyrille RATHGEBER - INRA Nancy
## ================================================================================================

plotCDC <- function(x, plot.type, hue) {
   
   # message("--> Entering plotCDC function...")

   ## Computing median, inter quartile range and upper and lower fence for outliers detection
   ## ---------------------------------------------------------------------------------------
   X <- sort(x)
   Y <- 1:length(X)
   L <- seq(0, max(X), 1)
   M <- median(X)
   Q <- quantile(X, probs = c(0.25, 0.75))
   IQ <- Q[2] - Q[1]
   UIF <- min(M + 1.5*IQ, max(X))
   UOF <- M + 3*IQ
   LIF <- max(max(M - 1.5*IQ, min(X)), 0)
   LOF <- max((M - 3*IQ), 0)
   
   ## Adjusting upper and lower fence in case of Cambial cells
   ## --------------------------------------------------------
   if (plot.type == "Cambial cells") LOF <- max((M - 3*IQ), 1)
  
   ## Drawing the plot region
   ## -----------------------
   plot(X, Y, type = "n", axes=FALSE, ann=FALSE)
   axis(1)
   axis(2)
   mtext(plot.type, side=3, line=0.1, adj=0)
   box()

   ## Defining the color code
   ## -----------------------
   Col.IFR <- hsv(h=hue, s=0.3, v=1, alpha=1)
   Col.IQR <- hsv(h=hue, s=0.5, v=0.9, alpha=1)
   Col.Med <- hsv(h=hue, s=0.7, v=0.5, alpha=1)
   Col.Data.Points <- hsv(h=hue, s=0.9, v=0.5, alpha=1)
  
   ## Plotting median, inter quartile range and upper and lower fence for outliers detection
   ## ---------------------------------------------------------------------------------------
   polygon(c(LIF, UIF, UIF, LIF), c(0, 0, max(Y), max(Y)), col=Col.IFR, border=Col.IFR)
   polygon(c(Q[1:2],Q[2:1]), c(0, 0, max(Y), max(Y)), col=Col.IQR, border=Col.IQR)
   abline(v = L, col = "grey")
   lines(c(M, M), c(0, max(Y)), col=Col.Med, lwd=2)
  
   ## Ploting the data points
   ## -----------------------
   points(X, Y, pch=16, col=Col.Data.Points)
  
   ## Marking of outliers
   ## -------------------
   points(X[X < LOF], Y[X < LOF], col = "red", pch = 1, cex=4)
   points(X[X > UOF], Y[X > UOF], col = "red", pch = 1, cex=4)

   ## Returning Upper and Lower outer fence values for subsequent use in outliers detection
   ## -------------------------------------------------------------------------------------
   OFV <- c(LOF, UOF)
   names(OFV) <- c("Lower.outer.fence", "Upper.outer.fence")
   return(OFV)
 
} # End Function plotCDC
## ************************************************************************************************