## ************************************************************************************************
## plotCleavelandDC() function definition
## --------------------------------------
##
## Internal function for plotting one Cleaveland Dot Chart per phase of development for a group of trees,
##    when previous ring measurement are not available or not of interest
## 
## Arguments:
##    - data: data.frame with imposed column names:
##       Site, Year, Tree, Species, Sample, DY, RF, CZ , EZ, WZ, MZ
##
## Outputs:
##		- A figure combining four Cleaveland Dot Charts for cell counts data
##    - A matrix containing the detected potential outliers for the different phases
##
## Versions:
##		1.0.0 Function developemnt started.
##    1.0-1. 2014-10-09 --- Fixing typos.
##    1.1.0. 2015-12-15 --- Changing plot title.
##    1.1.1. 2016-06-22 --- Changing variable names.
##    1.1.2. 2016-06-24 --- Handling case where there is no data to plot.
##    1.1.3. 2016-06-24 --- Including zero and negative values in outliners.
##
## Started: 22 Jully 2014
## Last modifications: 24 June 2016
## Author: Cyrille RATHGEBER - INRA Nancy
## ================================================================================================

plotCleavelandDC <- function(data) {
   
   # message("--> Entering plotCleavelandDC function...")
   DF <- data
   
   ## Setting the figure Layout for the 4 plots
   layout(rbind(c(1, 2), c(3, 3), c(4, 4), c(5, 5)),
          heights=c(1.25, 0.98, 0.98, 1.05),
          widths=c(7, 3))
   
   ## Setting the figure title
   main <- paste("Site: ", DF$Site[1], " - Year: ", DF$Year[1], " - Species: ", DF$Species[1])
   
   
   ## Plotting Cleaveland dot chart for cambial cells
   ## ===============================================
   ## Setting plot parameters
   par(mar=c(3, 4, 4, 1))
   
   ## Checking that CZ data are available
   ## -----------------------------------
   NoDataToPlot <- FALSE
   DF.NAO <- na.omit(DF[c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "CZ")])
   if (nrow(DF.NAO) == 0) NoDataToPlot <- TRUE

   ## Calling plotCDC or plotND internal functions
   ## --------------------------------------------
   COL <- data.frame()
   if (NoDataToPlot == FALSE) {
      
      ## Calling plotCDC internal function
      COF <- plotCDC(DF$CZ, plot.type="Cambial cells", hue=0.3)
      
      ## Extracting detected outliers
      COL <- na.omit(DF[DF$CZ < COF[1] | DF$CZ > COF[2],
                        c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "CZ")])
   } else {
      
      ## Calling plotEmptyCDC internal function
      plotEmptyCDC(DF$CZ, plot.type="Cambial cells")
   }
   
   ## Adding a title to the figure
   mtext(main, side=3, line=2, cex=1.25, font=2, adj=0)
   
   
   ## Writing the legend in the second panel
   ## ======================================
   ## Setting plot parameters
   par(mar=c(2, 2, 4, 1))
   
   ## Definig a plot region
   X <- 0:10
   Y <- 0:10
   #plot(X, Y, type = "n")
   plot(X, Y, type = "n", axes=FALSE, ann=FALSE)
   
   # Writing Legend title
   mtext("Legend", side=3, line=0.5, adj=0.04)
   
   # Writing Legend for data points and outliers
   points(1, 9, col = "grey25", pch = 16)
   points(1, 9, col = "red", pch = 1, cex=4)
   text(2, 9.5, labels="Data point...", pos=4, col = "grey25")
   text(2.5, 8.5, labels="Marked as an outlier", pos=4, col="red")
   
   ## Writing the Legend for median, interquartile range, and upper and lower fence
   polygon(c(0, 2, 2, 0), c(1, 1, 7, 7), col="grey75", border="grey75")
   polygon(c(0, 2, 2, 0), c(3, 3, 5, 5), col="grey50", border="grey50")
   lines(c(0, 2), c(4, 4), col="grey25", lwd=2)
   text(3, 6.5, labels="Outer fence", pos=4, col="grey75")
   text(2, 4, labels="Median", pos=4, col = "grey25")
   text(2.5, 2.5, labels="Inter quartile", pos=4, col="grey50")
   
   
   ## Plotting Cleaveland dot chart for enlarging cells
   ## =================================================
   ## Setting plot parameters
   par(mar=c(3, 4, 0, 1))
   
   ## Checking that EZ data are available
   ## -----------------------------------
   NoDataToPlot <- FALSE
   DF.NAO <- na.omit(DF[c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "EZ")])
   if (nrow(DF.NAO) == 0) NoDataToPlot <- TRUE
   
   ## Calling plotCDC or plotND internal functions
   ## --------------------------------------------
   EOL <- data.frame()
   if (NoDataToPlot == FALSE) {
      
      ## Calling plotCDC internal function
      EOF <- plotCDC(DF$EZ, plot.type="Enlarging cells", hue=0.6)
      
      ## Extracting detected outliers
      EOL <- na.omit(DF[DF$EZ < EOF[1] | DF$EZ > EOF[2],
                        c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "EZ")])
   } else {
      
      ## Calling plotEmptyCDC internal function
      plotEmptyCDC(DF$EZ, plot.type="Enlarging cells")
   }
   
   
   ## Plotting Cleaveland dot chart for wall thickening cells
   ## =======================================================
   ## Setting plot parameters
   par(mar=c(3, 4, 0, 1))
   
   ## Checking that WZ data are available
   ## -----------------------------------
   NoDataToPlot <- FALSE
   DF.NAO <- na.omit(DF[c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "WZ")])
   if (nrow(DF.NAO) == 0) NoDataToPlot <- TRUE
   
   ## Calling plotCDC or plotND internal functions
   ## --------------------------------------------
   WOL <- data.frame()
   if (NoDataToPlot == FALSE) {
      
      ## Calling plotCDC internal function
      WOF <- plotCDC(DF$WZ, plot.type="Thickening cells", hue=0)
      
      ## Extracting detected outliers
      WOL <- na.omit(DF[DF$WZ < WOF[1] | DF$WZ > WOF[2],
                        c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "WZ")])
   } else {
      
      ## Calling plotEmptyCDC internal function
      plotEmptyCDC(DF$WZ, plot.type="Thickening cells")
   }
   
   ## Customising axes 2
   mtext("Index of ordered observations", side=2, line=2.5, adj=-0.5)
   
   
   ## Plotting Cleaveland dot chart for mature cells
   ## ==============================================
   ## Setting plot parameters
   par(mar=c(4, 4, 0, 1))
   
   ## Checking that MZ data are available
   ## -----------------------------------
   NoDataToPlot <- FALSE
   DF.NAO <- na.omit(DF[c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "MZ")])
   if (nrow(DF.NAO) == 0) NoDataToPlot <- TRUE
   
   ## Calling plotCDC or plotND internal functions
   ## --------------------------------------------
   MOL <- data.frame()
   if (NoDataToPlot == FALSE) {
      
      ## Calling plotCDC internal function
      MOF <- plotCDC(DF$MZ, plot.type="Mature cells", hue=0.8)
      
      ## Extracting detected outliers
      MOL <- na.omit(DF[DF$MZ < MOF[1] | DF$MZ > MOF[2],
                        c("Site", "Year", "Tree", "Species", "Sample",  "DY", "RF", "MZ")])
   } else {
      
      ## Calling plotEmptyCDC internal function
      plotEmptyCDC(DF$MZ, plot.type="Mature cells")
   }
   
   ## Customising axes 1
   mtext("Number of cells", side=1, line=2.5)
   
   
   ## Adding CaviaR stamp on the figure
   ## =================================
   mtext("CaviaR.2 - plotCleavelandDotChart function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
   
   
   ## Returning outliers values for all the phases
   ## ============================================
   ODF <- list(Cambial.cell.outliers=COL, Enlarging.cell.outliers=EOL, Thickening.cell.outliers=WOL,
               Mature.cell.outliers=MOL)
   return(ODF)
   
   
} # End plotCleavelandDC
## ************************************************************************************************