## ************************************************************************************************
## plotCleavelandDotChart() function definition
## --------------------------------------------
##
## External function for plotting one Cleaveland Dot Chart per phase of development,
##    and per group of trees for a whole data set
## 
## Arguments:
##    - data: a data.frame with imposed column names:
##       Site, Year, Tree, Species, Sample, DY, RF, CZ , EZ, WZ, MZ, and optionally PR
##    - show.previous.ring (optional): a logical indicating if previous ring measurements
##       must be used (default=TRUE) or not (FALSE)
##
## Output:
##    - A figure combining several Cleaveland Dot Charts for cell counts data
##    - A List containing the detected potential outliers for the different phases of development
##       for the different group of trees
##
## Versions:
##	 1.0-0. Function developemnt started.
##    1.0-1. 2014-10-09 --- Fixing typos.
##    1.1.0. 2015-12-15 --- Changing plot title.
##    1.1.1. 2016-06-22 --- Changing variable names.
##    1.1.2. 2016-06-24 --- Handling case where there is no data to plot.
##    1.1.3. 2016-06-24 --- Including zero and negative values in outliners.
##    1.1.4. 2016-06-27 --- Changing variable names again.
##
## Started: 22 Jully 2014
## Last modifications: 27 June 2016
## Author: Cyrille RATHGEBER - INRA Nancy
## ================================================================================================

plotCleavelandDotChart <- function(data, show.previous.ring = TRUE) {
   
   # message("--> Entering plotCleavelandDotChart function...")
   IDF <- data
   
   ## Checking that all the needed factors and variables are included
   ## ===============================================================
   ## Checking that all the needed factors are included
   ## -------------------------------------------------
   if ("Site" %in% names(IDF) == FALSE) stop("Site column is missing!")
   if ("Year" %in% names(IDF) == FALSE) stop("Year column is missing!")
   if ("Species" %in% names(IDF) == FALSE) stop("Species column is missing!")
   
   ## Checking that all the needed variables are included
   ## ---------------------------------------------------
   if ("DY" %in% names(IDF) == FALSE) stop("DY column is missing!")
   if ("CZ" %in% names(IDF) == FALSE) stop("CZ column is missing!")
   if ("EZ" %in% names(IDF) == FALSE) stop("EZ column is missing!")
   if ("WZ" %in% names(IDF) == FALSE) stop("WZ column is missing!")
   if ("MZ" %in% names(IDF) == FALSE) stop("MZ column is missing!")  
   
   
   ## Selecting which form of the function to use depending on the option
   ## ===================================================================
   ## If option show.previous.ring = FALSE
   ## --> Previous ring measurements are not available or not of interest
   ## -------------------------------------------------------------------
   if (show.previous.ring == FALSE) {
      
      ## Call plotCleavelandDC internal function to draw Cleaveland dot plot for a group of trees
      ## This figure contains 4 plots (C, E, W, M)
      OLST <- by(IDF, list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species), plotCleavelandDC)
      
   } ## End if (show.previous.ring == FALSE)

   ## If option show.previous.ring = TRUE
   ## --> Previous ring measurements are available and of interest
   ## ------------------------------------------------------------
   if (show.previous.ring == TRUE) {
      
      ## Checking that PR variable is included
      if ("PR" %in% names(IDF) == FALSE) stop("Previous ring column (PR) is missing!")
      
      ## Call plotCleavelandDC internal function to draw Cleaveland dot plot for a group of trees
      ## This figure contains 5 plots (C, E, W, M, P)
      OLST <- by(IDF, list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species), plotCleavelandDC.PR)
      
   } ## End if (show.previous.ring == TRUE)
   
   return(OLST)
   
} ## End plotCleavelandDotChart function
## ************************************************************************************************