## *********************************************************************************************************
## plotGompertz.CT function definition
## -----------------------------------
##
## Purpose: Visualisation function for plotting the mean Gompertz curves for a group of trees
##
## Arguments:
##   - gompertz.simulations: a data frame containing Gompertz model simulations for all the day of the year.
##		This is typically the output of fitGompertzModels function.
##   - hue: a numeric indicating hue value (default = 0.6).
##   - plot.title: a character string indicating the title of the plot
##
## Output:
##   - One plot with the one Gompertz curve fitted on cell count data (EWMZ),
##      showing the mean of the group and its standard error enveloppe.
##
## Versions:
##		1.1-0. Defining this new function (21/11/2016)
##		1.1-1. Checking this new function (29/11/2016)
##    2.1-1. Improvement and homogeneisation (12/04/2017)
##
## Started: 21 November 2016
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

plotGompertz.CT <- function(gompertz.simulations, hue=0.6, plot.title=NULL) {
   
   #message("--> Entering plotGompertz.CT function...")
   
   ## Data manipulation
   ## *****************
   ## Reading input data
   ## ------------------
   IDF <- gompertz.simulations
   
   ## Computing mean standard error manually (because nls option se.fit is ignored!)
   ## ==============================================================================
   ## Defining a function to compute the standard error of the mean
   ## -------------------------------------------------------------
   sem <- function(x) {
      sem <- sd(x, na.rm=TRUE) / sqrt(length(x))
   }
   
   ## Computing the mean and the mean standard error
   ## ----------------------------------------------
   ## Computing the mean
   MNDF <- aggregate(IDF$EWMZ, by=list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species, DY=IDF$DY), mean)
   names(MNDF) <- c("Site", "Year", "Species", "DY", "Mean")
   
   ## Computing the sem
   SEDF <- aggregate(IDF$EWMZ, by=list(Site=IDF$Site, Year=IDF$Year, Species=IDF$Species, DY=IDF$DY), sem)
   names(SEDF) <- c("Site", "Year", "Species", "DY", "SEM")
   
   ## Creating a new data frame with the mean and the sem
   GEDF <- merge(MNDF, SEDF)
   GEDF <- GEDF[order(GEDF$DY), ]
   
   ## Computing high and low bound for the enveloppe
   GEDF$HB <- GEDF$Mean + GEDF$SEM
   GEDF$LB <- GEDF$Mean - GEDF$SEM
   
   
   ## Fitting new Gompertz functions on the mean and high and low bounds
   ## ====================================================================
   ## Fitting a Gompertz function for the mean
   ## ----------------------------------------
   R.fitGPZ.mean <- fitGPZ(GEDF$DY, GEDF$Mean)
   
   ## Recording the timings for the mean Gompertz curve
   am <- R.fitGPZ.mean$coef["a"]
   tim <- R.fitGPZ.mean$coef["ti"]
   tfm <- R.fitGPZ.mean$coef["tf"]
   txm <- R.fitGPZ.mean$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.mean <- R.fitGPZ.mean$sim
   
   ## Fitting a Gompertz function for the high bound
   ## ----------------------------------------------
   R.fitGPZ.hb <- fitGPZ(GEDF$DY, GEDF$HB)
   
   ## Recording the timings for the high Gompertz curve
   ahb <- R.fitGPZ.hb$coef["a"]
   tihb <- R.fitGPZ.hb$coef["ti"]
   tfhb <- R.fitGPZ.hb$coef["tf"]
   txhb <- R.fitGPZ.hb$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.hb <- R.fitGPZ.hb$sim
   
   ## Fitting a Gompertz function for the low bound
   ## ---------------------------------------------
   R.fitGPZ.lb <- fitGPZ(GEDF$DY, GEDF$LB)
   
   ## Recording the timings for the low Gompertz curve
   alb <- R.fitGPZ.lb$coef["a"]
   tilb <- R.fitGPZ.lb$coef["ti"]
   tflb <- R.fitGPZ.lb$coef["tf"]
   txlb <- R.fitGPZ.lb$coef["tx"]
   
   ## Storing the simulations in a convenient data frame
   SIMDF.lb <- R.fitGPZ.lb$sim

   ## Creating a new dataframe containing mean and high and low bounds
   ## ----------------------------------------------------------------
   SIMDF <- data.frame(DY=SIMDF.mean$DY, Mean=SIMDF.mean$Sim, HB=SIMDF.hb$Sim, LB=SIMDF.lb$Sim)
   
   
   ## Plotting the data
   ## *****************
   ## Defining color code
   ## -------------------
   Col.env <- hsv(h=hue, s=0.3, v=0.9, alpha=1) ## Color for the enveloppe
   Col.cur <- hsv(h=hue, s=0.6, v=0.8, alpha=1) ## Color for the mean curve
   Col.pts <- hsv(h=hue, s=0.9, v=0.7, alpha=1) ## Color for the points and timing bars
   
   ## Defining maximal and minimal values for the plot
   ## ------------------------------------------------
   xmin <- max(c(0, tihb - tihb/5))
   xmax <- min(c(360, tfhb + tfhb/5))
   ymin <- 0
   ymax <- ahb + ahb/10
   
   ## Defining the plot region
   ## ------------------------
   plot(1, 1, xlim=c(xmin, xmax), ylim=c(ymin, ymax), ann=FALSE, axes=FALSE)
   
   ## Drawing the enveloppe of the mean Gompertz curve
   ## ------------------------------------------------
   ## Creating a new data frame for plotting the polygon
   PLGDF <- data.frame(X=c(SIMDF$DY, SIMDF$DY[rev(order(SIMDF$DY))]),
                       Y=c(SIMDF$LB, SIMDF$HB[rev(order(SIMDF$HB))]))
   
   ## Drawing the enveloppe
   polygon(PLGDF$X, PLGDF$Y, col=Col.env, border=NA)
   
   ## Plotting the Gompertz curve for the mean of the group
   ## =====================================================   
   ## Plotting the curve
   lines(SIMDF$DY, SIMDF$Mean, col=Col.cur, lwd=2, lty=1)
   
   ## Plotting additional informations
   ## ================================
   ## Plotting the timings on the curve
   ## ---------------------------------
   points(tim, 0.05*am, type="p", pch=17, col=Col.pts, cex=1.5)
   points(txm, am/exp(1), type="p", pch=16, col=Col.pts, cex=1.5)
   points(tfm, 0.95*am, type="p", pch=15, col=Col.pts, cex=1.5)
   
   ## Drawing the timmings on the x-axis
   ## ----------------------------------
   rug(seq(from=min(tihb, tilb), to=max(tihb, tilb), by=0.25), col=Col.env, lwd=1)
   rug(tim, col=Col.pts, lwd=2)
   rug(seq(from=min(txhb, txlb), to=max(txhb, txlb), by=0.25), col=Col.env, lwd=1)
   rug(txm, col=Col.pts, lwd=2)
   rug(seq(from=min(tfhb, tflb), to=max(tfhb, tflb), by=0.25), col=Col.env, lwd=1)
   rug(tfm, col=Col.pts, lwd=2)
   
   ## Writing labels for the timing signs
   ## -----------------------------------
   mtext(expression(alpha), side=4, line=0.25, at=am, col=Col.pts, las=1)
   mtext(expression(t[b]), side=1, line=0.25, at=tim, col=Col.pts)
   mtext(expression(t[x]), side=1, line=0.25, at=txm, col=Col.pts)
   mtext(expression(t[c]), side=1, line=0.25, at=tfm, col=Col.pts)
   
   ## Drawing axes
   ## ------------
   axis(1)
   axis(2)
   box()
   
   ## Writting the title
   ## ------------------
   if (is.null(plot.title)) {
      plot.title <- paste("Site: ", IDF$Site[1], " - ", 
                     "Year: ", IDF$Year[1], " - ",
                     "Species: ", IDF$Species[1])
   }
   mtext(plot.title, side=3, line=1, cex=1.25, font=2)
   
   ## Writting the labels of the axes
   ## -------------------------------
   mtext("Day of Year", side=1, line=2.5)
   mtext("Cumulated number of cells", side=2, line=2.5)
   
   ## Adding CaviaR stamp on the figure
   ## ---------------------------------
   mtext("CaviaR.2 - plotWoodGrowth function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
   
   ## Returning verbose output
   return("--> Done")
   
} ## End function plotGompertz.CT

## ------------------------------------------------------------------------------------------------
##                           End plotGompertz.CT function
## ************************************************************************************************