## *********************************************************************************************************
## plotGompertz.Comp function definition
## -----------------------------------
##
## Purpose: Visualisation function for comparing the mean Gompertz curves for a sevral group of trees
##
## Arguments:
##   - gompertz.simulations: a data frame containing Gompertz model simulations for all the day of the year.
##		This is typically the output of fitGompertzModels function.
##   - plot.by: a character string indicating which class must be represented on the plot (default = "Species")
##   - hues: a numeric vector containing a hue value for each group to define color palettes.
##   - plot.title: a character string containing the title of the plot.
##
## Output:
##   - One plot with the sevral Gompertz curves fitted on cell count data (EWMZ),
##      showing the mean of the groups and their standard error enveloppes.
##
## Versions:
##		1.1-0. Defining this new function (30/11/2016)
##		1.1-1. Checking this new function (01/12/2016)
##		2.1-0. Improving this function in order to take more than 3 groups (13/01/2017)
##    2.1-1. Improvement and homogeneisation (12/04/2017)
##
## Started: 30 November 2016
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ---------------------------------------------------------------------------------------------------------

plotGompertz.Comp <- function(gompertz.simulations, plot.by = "Species", hues = NULL, plot.title = NULL) {
   
   #message("--> Entering plotGompertz.Comp function...")
   
   ## Data manipulation and plot settings
   ## ***********************************
   ## Reading and checking input data
   ## -------------------------------
   IDF <- gompertz.simulations

   ## Defining maximal and minimal values for the plot - using rules of thumb!
   ## ------------------------------------------------------------------------
   ymin <- 0
   ymax <- 0.9 * max(IDF$EWMZ)
   xmin <- max(50, subset(IDF, IDF$EWMZ > 0.1, select = DY)$DY[1], na.rm = TRUE)
   xmax <- min(c(350, subset(IDF, IDF$EWMZ > 0.99*max(IDF$EWMZ), select = DY)$DY[1]), na.rm = TRUE)
   
   ## Defining the color palette
   ## --------------------------
   group.index <- as.character(IDF[c(plot.by)][, 1])
   f.group <- as.factor(group.index)
   n.group <- as.integer(nlevels(f.group))
   if(is.null(hues)) {
      hues=seq(from=0.1, to=0.9, length.out=n.group)
      names(hues) <- levels(f.group)
   }

   
   ## Plotting the data
   ## *****************
   ## Defining the plot region
   plot(1, 1, xlim=c(xmin, xmax), ylim=c(ymin, ymax), ann=FALSE, axes=FALSE)
   
   ## Caling drawSmartGompertzCurve.IUF for each group of trees
   ## =========================================================
   cnt <- 0
   for (grp in levels(f.group)) {

      ## Defining the color code
      ## -----------------------
      hue <- hues[grp]
      Col.env <- hsv(h=hue, s=0.3, v=0.9, alpha=0.75) ## Color for the enveloppe
      Col.cur <- hsv(h=hue, s=0.6, v=0.8, alpha=1) ## Color for the mean curve
      Col.pts <- hsv(h=hue, s=0.9, v=0.7, alpha=1) ## Color for the points and timing bars
      Colors <- c(Col.env, Col.cur, Col.pts)
      
      ## Caling drawSmartGompertzCurve.IUF
      drawSmartGompertzCurve.IUF(IDF[group.index==grp, ], colors=Colors)
      
      ## Drawing legend element
      ## ----------------------
      cnt <- cnt + 1
      X1 <- xmin + 2
      X2 <- X1 + 10
      Xt <- X2 + 1
      Yh <- (ymax - 2) - (7 * (n.group - cnt))
      Yt <- Yh - 3
      Yl <- Yh - 6
      lines(c(X1, X2), c(Yt, Yt), col=Col.cur, lwd=2)
      polygon(c(X1, X2, X2, X1), c(Yl, Yl, Yh, Yh), col=Col.env, border=NA)
      points((X1 + X2)/2, Yt, type="p", pch=16, col=Col.pts, cex=1.5)
      text(Xt, Yt, labels=grp, col=Col.pts, pos=4)
      
   } ## End for (grp in levels(as.factor(f.group)))

   ## Drawing axes
   ## ------------
   axis(1)
   axis(2)
   box()
   
   ## Writting the title
   ## ------------------
   plot.title.site <- paste(unique(IDF$Site), collapse = ", ")
   plot.title.year <- paste(unique(IDF$Year), collapse = ", ")
   plot.title.species <- paste(unique(IDF$Species), collapse = ", ")
   
   if (is.null(plot.title)) {
      plot.title <- paste("Site: ", plot.title.site, " - ", 
                     "Year: ", plot.title.year, " - ",
                     "Species: ", plot.title.species)
   }
   mtext(plot.title, side=3, line=1, cex=1, font=2)
   
   ## Writting the labels of the axes
   ## -------------------------------
   mtext("Day of Year", side=1, line=2.5)
   mtext("Cumulated number of cells", side=2, line=2.5)
   
   ## Adding CaviaR stamp on the figure
   ## ---------------------------------
   mtext("CaviaR.2 - plotWoodGrowth function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
   
   ## Returning verbose output
   return("--> Done")
   
} ## End function plotGompertz.Comp

## ------------------------------------------------------------------------------------------------
##                           End plotGompertz.CT function
## ************************************************************************************************