###################################################################################################
## plotGompertz.Group function definition
## ======================================
##
## Purpose: Visualisation function for plotting Gompertz curves for all the trees of a group
##
## Arguments:
##   - data: a data frame containing seven columns composed of:
##    (1) 4 grouping variables (Site, Year, Species, Tree);
##    (2) 3 parameters of the Gompertz (a, b, k)
##    (3) 3 biological parameters (ti, tf, tx)
##		This is typically modified from the output of fitGompertzModels function.
##
## Output:
##   - One plot with the Gompertz curves fitted on cell count data (EWMZ),
##      showing all the trees from the group.
##
## Versions:
##		1.1-0. Defining this new function (16/11/2016)
##		1.1-1. Checking this new function (21/11/2016)
##		1.1-1. Extending its use to 15 trees (12/01/2017)
##    1.1-2 Improvement and homogeneisation (12/04/2017)
##
## Started: 16 November 2016
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - INRA Nancy
##
###################################################################################################

plotGompertz.Group <- function(Gompertz.parameters.group) {
   
   #message("--> Entering plotGompertz.Group function...")
   
   ## reading input data
   ## ------------------
   IDF <- Gompertz.parameters.group
   
   ## Plotting data
   ## =============
   ## Defining maximal values for the plot
   ## ------------------------------------
   xmin <- max(c(0, min(IDF$ti) - min(IDF$ti)/10))
   xmax <- min(c(360, max(IDF$tf) + 2*max(IDF$tf)/10))
   ymin <- 0
   ymax <- max(IDF$a) + max(IDF$a)/10
   
   ## Defining the color scale
   ## ------------------------
   NoT <- nrow(IDF)
   if (NoT <= 5) {
      SoC <- c("skyblue", "deepskyblue", "dodgerblue", "steelblue", "navy")    
   } else {
      SoC <- sample(rainbow(NoT), size=NoT)     
   }

   ## Drawing the plot region
   ## -----------------------
   plot(1, 1, xlim=c(xmin, xmax), ylim=c(ymin, ymax), ann=FALSE, axes=FALSE)
      
   ## Plotting Gompertz curves for all the trees of the group
   ## -------------------------------------------------------
   for (i in 1:nrow(IDF)) {
      tree.id <- IDF$Tree[i]
      ai <- IDF$a[i]
      bi <- IDF$b[i]
      ki <- IDF$k[i]
      tii <- IDF$ti[i]
      tfi <- IDF$tf[i]
      txi <- IDF$tx[i]
      GFi <- function(x) ai*exp(-exp(bi - ki*x))
      ## Plotting the curve
      curve(GFi, from=0, to=360, add=TRUE, lwd=1, col=SoC[i])
      ## Plotting the assymptote
      abline(h=ai, col=SoC[i], lty=2)
      ## Plotting the timings
      points(tii, 0.05*ai, type="p", pch=24, cex=1.5, col=SoC[i])
      points(txi, ai/exp(1), type="p", pch=21, cex=1.5, col=SoC[i])
      points(tfi, 0.95*ai, type="p", pch=22, cex=1.5, col=SoC[i])
      ## Labelling the curve
      text(xmax, ai, labels=tree.id, pos=3, font=1, col=SoC[i])
   }
   
   ## Plotting Gompertz curves for the mean of the group
   ## --------------------------------------------------
   ## computing the mean parameters for the group
   Param <- apply(IDF[, 5:10], MARGIN=2, FUN=mean, na.rm=TRUE)
   am <- Param["a"]
   bm <- Param["b"]
   km <- Param["k"]
   tim <- Param["ti"]
   tfm <- Param["tf"]
   txm <- Param["tx"]
   colGm <- "royalblue"
   
   ## Defining Gompertz function
   GFm <- function(x) am*exp(-exp(bm - km*x))
   ## Plotting the curve
   curve(GFm, from=0, to=360, add=TRUE, col=colGm, lwd=2, lty=1)
   ## Plotting the assymptote
   abline(h=am, col=colGm, lty=2)
   ## Plotting the timings
   points(tim, 0.05*am, type="p", pch=17, cex=1.5, col=colGm)
   points(txm, am/exp(1), type="p", pch=16, cex=1.5 , col=colGm)
   points(tfm, 0.95*am, type="p", pch=15, cex=1.5, col=colGm)
   ## Labelling the curve
   text(xmax - 20, am, labels="Mean", pos=3, font=1, col=colGm)
   
   ## Drawing rug
   ## ------------
   rug(jitter(IDF$ti), col="skyblue")
   rug(tim, col="skyblue", lwd=2)
   rug(jitter(IDF$tx), col="dodgerblue")
   rug(txm, col="dodgerblue", lwd=2)
   rug(jitter(IDF$tf), col="navy")
   rug(tfm, col="navy", lwd=2)
   
   ## Drawing axes
   ## ------------
   axis(1)
   axis(2)
   box()
      
   ## Title and axes labels
   ## ---------------------
   Title <- paste("Site: ", IDF$Site[1], " - ",
                  "Year: ", IDF$Year[1], " - ",
                  "Species: ", IDF$Species[1])
   mtext(Title, side=3, line=1, cex=1.25, font=2)
   mtext("Day of Year", side=1, line=2.5)
   mtext("Cumulated number of cells", side=2, line=2.5)
   
   ## Additional labels for the timing signs
   ## --------------------------------------
   mtext(expression(alpha), side=4, line=0.25, at=am, col=colGm, las=1)
   mtext(expression(t[b]), side=1, line=0.25, at=tim, col="skyblue")
   mtext(expression(t[x]), side=1, line=0.25, at=txm, col="dodgerblue")
   mtext(expression(t[c]), side=1, line=0.25, at=tfm, col="navy")
      
   ## Adding CaviaR stamp on the figure
   mtext("CaviaR.2 - plotWoodGrowth function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
      
   ## Returning verbose output
   return("--> Done")

} ## End function plotGompertz.Group

## ------------------------------------------------------------------------------------------------
##                           End plotGompertz.Group function
## ************************************************************************************************