## ************************************************************************************************
## plotGompertz.Tree function definition
## -------------------------------------
##
## Purpose: Visualisation function for plotting Gompertz curves with cumulative cell count data
##           for one tree
##
## Arguments:
##   - ILST: a list containing four data frames:
##    (1) the references of the group (references);
##    (2) the fitted coefficients (coefficients);
##    (3) the values of observations and model predictions (values); and
##    (4) the model simulations for every day of the year.
##		This is typically the output of fitGompertzModels function.
##
## Output:
##   - Three plots of observed cell count data (EWMZ, WMZ, MZ),
##     along with the three fitted Gompertz curves.
##
## Versions:
##   1. Plots of observed cell count data (EWMZ, WMZ, MZ),
##      along with the three fitted Gompertz curves.
##   2.1-1 General cleaning (07/11/2016)
##   2.1-2 Improvement and homogeneisation (12/04/2017)
##
## Started: 11 June 2012
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotGompertz.Tree <- function(Gompertz.parameters.tree) {

   #message("--> Entering plotGompertz.Tree function...")
   
   ## Preparing data
   ## ==============
   ## reading input list
   ILST <- Gompertz.parameters.tree
   
   ## Formating incoming data
	Ref <- ILST$references
	Coef <- as.data.frame(ILST$coefficients)
	Val <- as.data.frame(ILST$values)
	
   ## Plotting data
   ## =============
   ## Defining maximal values for the plot
   xmin <- min(Val$DY, na.rm=TRUE)
   xmax <- max(Val$DY, na.rm=TRUE)
   ymin <- 0
   ymax <- max(Val$EWMZ.obs, Val$WMZ.obs, Val$MZ.obs, na.rm=TRUE)
   
   ## Plotting signs for raw data
   plot(Val$DY, Val$EWMZ.obs, xlim=c(xmin, xmax), ylim=c(ymin, ymax),
   	  type="p", pch=3, col="blue", ann=FALSE, axes=FALSE)
   points(Val$DY, Val$WMZ.obs, type="p", pch=4, col="red")
   points(Val$DY, Val$MZ.obs, type="p", pch=1, col="purple")
   
   ## Plotting Growth curves
   ## ----------------------
   ## EWMZ curve
   C1 <- Coef[Coef$Type=="EWMZ", ]
   a1 <- C1$a
   b1 <- C1$b
   k1 <- C1$k
   GF1 <- function(x) a1*exp(-exp(b1 - k1*x))
   curve(GF1, from=0, to=360, add=TRUE, col="blue")
   
   ## WMZ curve
   C2 <- Coef[Coef$Type=="WMZ", ]
   a2 <- C2$a
   b2 <- C2$b
   k2 <- C2$k
   GF2 <- function(x) a2*exp(-exp(b2 - k2*x))
   curve(GF2, from=0, to=360, add=TRUE, col="red")
   
   ## MZ curve
   C3 <- Coef[Coef$Type=="MZ", ]
   a3 <- C3$a
   b3 <- C3$b
   k3 <- C3$k
   GF3 <- function(x) a3*exp(-exp(b3 - k3*x))
   curve(GF3, from=0, to=360, add=TRUE, col="purple")
   
   ## Plotting asymptotes
   ## -------------------
   ## EWMZ
   abline(h=a1, col="blue", lty=2)
   	
   ## WMZ
   abline(h=a2, col="red", lty=2)
   	
   ## WMZ
   abline(h=a3, col="purple", lty=2)
   	
   ## Labelling the M asymptote
   mtext(expression(alpha), side=4, line=0.25, at=a1, col="purple", las=1)
   	
   ## Drawing axes
   ## ------------
   axis(1)
   axis(2)
   box()
   
   ## Title and axes labels
   ## ---------------------
   Title <- paste("Site: ", Ref["Site"], " - ",
                  "Year: ", Ref["Year"], " - ",
                  "Species: ", Ref["Species"], " - ",
                  "Tree ", Ref["Tree"], sep="")
   mtext(Title, side=3, line=1, adj=0, font=2)
   mtext("Day of Year", side=1, line=2.5)
   mtext("Cumulated number of cells", side=2, line=2.5)
   
   ## Legend
   ## ------
   ## EWMZ
   x <- xmax - xmax/10
   X <- c(x, x + 10)
   y1 <- ymax/7
   Y1 <- c(y1, y1)
   lines(X, Y1, col="blue")
   points(x+5, y1, pch=3, col="blue")
   text(x+12, y1, labels=expression(n[EWM]), adj=0, font=1)
   
   ## WMZ
   y2 <- y1 - y1/3
   Y2 <- c(y2, y2)
   lines(X, Y2, col="red")
   points(x+5, y2, pch=4, col="red")
   text(x+12, y2, labels=expression(n[WM]), adj=0, font=1)
   
   ## MZ
   y3 <- y2 - y1/3
   Y3 <- c(y3, y3)
   lines(X, Y3, col="purple")
   points(x+5, y3, pch=1, col="purple")
   text(x+12, y3, labels=expression(n[M]), adj=0, font=1)
   
   ## Additional information
   ## ----------------------
   mtext(expression(EF[EWM]), side=3, line=-2, adj=0.05)
   mtext(paste(" = ", round(C1$EF, digits=2)), side=3, line=-2, adj=0.15)
   mtext(expression(EF[WM]), side=3, line=-3, adj=0.05)
   mtext(paste(" = ", round(C2$EF, digits=2)), side=3, line=-3, adj=0.15)
   mtext(expression(EF[M]), side=3, line=-4, adj=0.05)
   mtext(paste(" = ", round(C3$EF, digits=2)), side=3, line=-4, adj=0.15)
   	
   ## Adding CaviaR stamp on the figure
   mtext("CaviaR.2 - plotWoodGrowth function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
   	
   ## Returning verbose output
   return(paste(Title, "--> Done"))

} ## End function plotGompertz.Tree

## ------------------------------------------------------------------------------------------------
##                           End plotGompertz.Tree function
## ************************************************************************************************