## ************************************************************************************************
## plotWFD.CT function definition
## ------------------------------
##
## Arguments:
##   - data: data.frame with imposed column names: Site, Year, Tree, Species, Sample, DY, RF, NC
##   - zone = Name of the inspected zone, for output labelling (default = "NA")
##   - ylab = label of the y axis (default ="Number of cells")
##   - hue = Value of hue (default = 0.6)
##
## Outputs:
##   - one plot of wood formation dynamics per tree
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.1-1. Changing the names of the input variables
##   	1.1-2. Changing the names of scalingYAxis.IUF function
##   	1.1-3. Improving this function + changing its name (6-7/04/2017)
##
## Started: 16 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.CT <- function(data, zone = "NA", ylab = "Number of cells", hue = 0.6, stat="mean") {
   
   # message("--> Entering plotWFD.CT function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Formating incoming data
   ## -----------------------
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Tree <- as.factor(DF$Tree)
   DF$Species  <- as.factor(DF$Species)
   DF$Sample <- as.factor(DF$Sample)
   DF$DY <- as.integer(DF$DY)
   DF$NC <- as.integer(DF$NC)
   DF$Zone <- zone
   DF$Dummy <- ifelse(is.na(DF$NC), 0, 1)


   ## ====================================================================
   ## Creating a database to collect central tendency data
   ## ----------------------------------------------------
   ## Option 1. stat="median"
   ## Computing median and its confidence interval
   ## --------------------------------------------
   QDF <- unique(DF[, c("Site", "Year", "Species", "Sample", "DY")])
   QDF$Q1 <- tapply(DF$NC, DF$DY, quantile, probs=0.25, na.rm=TRUE) ## First quartile
   QDF$Q2 <- tapply(DF$NC, DF$DY, median, na.rm=TRUE) ## Median
   QDF$Q3 <- tapply(DF$NC, DF$DY, quantile, probs=0.75, na.rm=TRUE) ## Third quartile
   QDF$NV <- tapply(DF$Dummy, DF$DY, sum, na.rm=TRUE) ## Number of values
   QDF$IQR <- QDF$Q3 - QDF$Q1 # Inter Quartile Range
   QDF$SE <-  (1.5 * QDF$IQR) / sqrt(QDF$NV) ## Standard Error
   QDF$LCIL <- QDF$Q2 - QDF$SE # Lower Confindence Interval Limit
   QDF$UCIL <- QDF$Q2 + QDF$SE # Upper Confindence Interval Limit
      
   CTDF <- QDF[, c("Site", "Year", "Species", "Sample", "DY", "Q2", "LCIL", "UCIL")]
   names(CTDF)  <- c("Site", "Year", "Species", "Sample", "DY", "M", "LCIL", "UCIL")

   ## Option 2. stat="mean"
   ## ---------------------
   if (stat == "mean") {

      ## Computing mean and its confidence interval
      ## ------------------------------------------
      PDF <- unique(DF[, c("Site", "Year", "Species", "Sample", "DY")])
      PDF$Mean <- tapply(DF$NC, DF$DY, mean, na.rm=TRUE) ## Mean
      PDF$SD <- tapply(DF$NC, DF$DY, sd, na.rm=TRUE) ## Sandard Deviation
      PDF$NV <- tapply(DF$Dummy, DF$DY, sum, na.rm=TRUE) ## Number of values
      PDF$SE <-  (qnorm(0.975) * PDF$SD) / sqrt(QDF$NV) ## Standard Error
      PDF$LCIL <- PDF$Mean - PDF$SE # Lower Confindence Interval Limit
      PDF$UCIL <- PDF$Mean + PDF$SE # Upper Confindence Interval Limit
      
      CTDF <- PDF[, c("Site", "Year", "Species", "Sample", "DY", "Mean", "LCIL", "UCIL")]
      names(CTDF)  <- c("Site", "Year", "Species", "Sample", "DY", "M", "LCIL", "UCIL")     
   }


   ## ====================================================================
   ## Plotting central tendency
   ## -------------------------

   ## Setting plot parameters
   ## -----------------------
   Ymax <- max(DF$NC, na.rm=TRUE)
   Scl <- Ymax / 10
   Ymin <- min(DF$NC, na.rm=TRUE)
   CEX <- 2

   ## Defining the plot region
   ## ------------------------
   plot(DF$DY, DF$NC, type="n",
        xlim=c(min(DF$DY), max(DF$DY)), ylim=c(0, Ymax + Scl),
        ann=FALSE, axes=FALSE)

   ## defining color code
   ## -------------------
   Col.CIE <- hsv(h=hue, s=0.5, v=0.9, alpha=1) ## Color for confidence interval enveloppe
   Col.M <- hsv(h=hue, s=0.9, v=0.5, alpha=1) ## Color for central tendency

   ## Plottind the confidence interval enveloppe for the central tendency
   ## -------------------------------------------------------------------
   CIE <- data.frame(X=c(CTDF$DY, CTDF$DY[rev(order(CTDF$DY))]), Y=c(CTDF$LCIL, CTDF$UCIL[rev(order(CTDF$DY))]))
   CIE$Y <- ifelse(CIE$Y < 0, 0, CIE$Y)
   CIE <- na.omit(CIE)
   polygon(CIE$X, CIE$Y, density=NA, col=Col.CIE)

   ## Drawing vertical lines for the sampling dates
   ## ---------------------------------------------
   abline(v=DF$DY, lty=1, col="grey90")

   ## Plotting the central tendency for the trees
   ## -------------------------------------------
   X.M <- CTDF$DY
   Y.M <- CTDF$M
   lines(X.M, Y.M, lwd=2, lty=1, col=Col.M)
   points(X.M, Y.M, pch=16, col=Col.M)

   ## Customising Axes
   ## ----------------
   ## Customising axes 1
   #abline(h=0)
   if (zone == "PR") {} else lines(c(min(DF$DY), max(DF$DY)), c(0, 0))

   ## Customising axes 2
   axis(2, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))
   mtext(ylab, side=2, line=2.5)
   
   ## Customising axes 4
   axis(4, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))

}  ## End plotWFD.CT

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.CT function
## ************************************************************************************************
