## ************************************************************************************************
## plotWFD.CTendency.PR function definition
## ----------------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics
##  - used for plotting wood formation dynamics for data at group level
##  - when previous ring measurements are not available
##
## Arguments:
##   - data: data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, PR
##
## Outputs:
##   - four plots for the four phases of differentiation by group of trees
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.0-1. Changing the names of the input variables
##   	1.1-3. Improving this function + changing its name (6-7/04/2017)
##
## Started: 23 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.CTendency.PR <- function(data, stat="mean") {
   
   # message("--> Entering plotWFD.CTendency.PR function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Additional data for plotting: month characteristics
   ## ---------------------------------------------------
   MonthId <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
   FirstDay <- c(1, 32, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335)
   MiddleDay <- c(16, 45, 75, 105, 136, 166, 197, 228, 258, 289, 319, 350)
   
   ## Setting the figure Layout for 5 plots
   layout(1:5, heights=c(1.3, 0.98, 0.98, 0.98, 1.3))
   
   ## Setting the figure title
   main <- paste("Site: ", DF$Site[1], " - Year: ", DF$Year[1], " - Species: ", DF$Species[1], sep="")
   
   
   ## Plotting cambial cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 4, 2))
   
   ## Calling plotWFD.CT
   ## ------------------
   CCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "CZ")]
   names(CCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.CC <- "Cambial cells"
   plotWFD.CT(CCDF, zone="CZ", hue=0.30, ylab=Ylab.CC, stat=stat)

   
   ## Customising axes 3
   ## ------------------
   axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
   axis(3, at=MiddleDay, tick=TRUE, labels=MonthId)
   
   ## Adding a title to the plot
   ## --------------------------
   mtext(main, side=3, line=2, cex=1.25, font=2)
   
   
   ## Plotting enlarging cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.CT
   ## ------------------
   ECDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "EZ")]
   names(ECDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.EC <- "Enlarging"
   plotWFD.CT(ECDF, zone="EZ", hue=0.60, ylab=Ylab.EC, stat=stat)
   
   
   ## Drawing the legend of the figure
   ## --------------------------------
   ## Computing Y scale
   ## -----------------
   Ymax <- max(DF$E, na.rm=TRUE)
   Yscl <- (Ymax) / 10

   ## Writing the Legend for central tendency and confidence interval
   ## ---------------------------------------------------------------
   wf <- 0.9
   LDY <- unique(DF$DY)
   XP <- LDY[length(LDY) - 3]
   XT <- LDY[length(LDY) - 2]
   X1  <- XP - wf*(XT - XP)
   X2  <- XP + wf*(XT - XP)
   YCI1 <- Ymax - 1*Yscl
   YM <- Ymax - 2*Yscl
   YCI2 <- Ymax - 3*Yscl
   polygon(c(X1, X2, X2, X1), c(YCI1, YCI1, YCI2, YCI2), col="grey70", border="grey70")
   lines(c(X1, X2), c(YM, YM), col="grey40", lty=1, lwd=2)
   points(XP, YM, pch=16, col="grey40")
   text(XT, YM, labels=stat, col = "grey40", adj=0)
   text(XT, YCI2, labels="CI", col="grey50", adj=0)

   
   ## Plotting thickening cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.CT
   ## ------------------
   WCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "WZ")]
   names(WCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.WC <- "Thickening"
   plotWFD.CT(WCDF, zone="WZ", hue=0.00, ylab=Ylab.WC, stat=stat)
   
   
   ## Plotting mature cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.CT
   ## ------------------
   MCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "MZ")]
   names(MCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.MC <- "Mature"
   plotWFD.CT(MCDF, zone="MZ", hue=0.80, ylab=Ylab.MC, stat=stat)


   ## Plotting previous ring measurements
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(4, 4, 0, 2))

   ## Calling plotWFD.CT
   ## ------------------
   PRDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "PR")]
   names(PRDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.PR <- "Previous ring"
   plotWFD.CT(PRDF, zone="PR", hue=0.15, ylab=Ylab.PR)

   ## Customising x axis of the figure
   ## --------------------------------
   axis(1, at=DF$DY, labels=FALSE)
   text(x=DF$DY, y=(par("usr")[3] - 3), labels=DF$DY, srt=45, pos=1, xpd=TRUE)
   mtext("Day of Year", side=1, line=2.5)

   
   ## Adding CaviaR stamp on the figure
   ## =================================
   mtext("CaviaR.2 - plotWoodFormationDynamics function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)

   
} # End plotWFD.CTendency.PR

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.CTendency.PR function
## ************************************************************************************************
