## ************************************************************************************************
## plotWFD.Comp function definition
## --------------------------------
##
## Arguments:
##   - data: data.frame with imposed column names: Site, Year, Tree, Species, Sample, DY, NC1, NC2
##   - zone = Name of the inspected zone, for output labelling (default = "NA")
##   - hue = Value of hue (default = 0.6)
##   - ylab = label of the y axis (default ="Number of cells")
##   - legend = labels for the legend (default = c("data 1", "data 2"))
##
## Outputs:
##   - one figurecomparing two estimation of wood formation dynamics per tree
##
## Versions:
##   	1.0-0. Defining this new function (10/10/2016)
##   	1.1-1. Improving this new function (11-18/10/2016)
##   	1.1-2. Improving this function + changing its name (6-7/04/2017)
##
## Started: 10 October 2016
## Last modifications: 7 April 2017
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.Comp <- function(data, zone = "NA", hue = 0.6, ylab = "Number of cells",
                         legend = c("raw", "stan")) {
   
   # message("--> Entering plotWFD.TD function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Formating incoming data
   ## -----------------------
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Tree <- as.factor(DF$Tree)
   DF$Species  <- as.factor(DF$Species)
   DF$DY <- as.integer(DF$DY)
   DF$NC1 <- as.numeric(DF$NC1)
   DF$NC2 <- as.numeric(DF$NC2)
   DF$Dif <- abs(DF$NC1 - DF$NC2)
   DF <- DF[order(DF$DY), ]


   ## Plotting cell count data
   ## ========================
   ## Setting plot parameters
   ## -----------------------
   Ymax <- max(c(DF$NC1, DF$NC2), na.rm=TRUE)
   Yscl <- Ymax / 10
   Ymin <- min(c(DF$NC1, DF$NC2), na.rm=TRUE)
   CEX <- 2

   ## Defining the plot region
   ## ------------------------
   plot(c(DF$DY, DF$DY), c(DF$NC1, DF$NC2),
        type="n", xlim=c(min(DF$DY), max(DF$DY)), ylim=c(0, Ymax + Yscl),
        ann=FALSE, axes=FALSE)

   ## defining color code
   ## -------------------
   Col1 <- hsv(h=hue, s=0.9, v=0.5, alpha=1) ## Color for data 1
   Col2 <- hsv(h=hue, s=0.8, v=0.7, alpha=1) ## Color for data 2
   Col3 <- hsv(h=hue, s=0.7, v=0.9, alpha=1) ## Color for difference bars

   ## Drawing vertical lines for the sampling dates
   ## ---------------------------------------------
   abline(v=DF$DY, lty=1, col="grey90")

   ## Plotting data
   ## -------------
   ## Data 1
   lines(DF$DY, DF$NC1, lwd=2, lty=1, col=Col1)
   points(DF$DY, DF$NC1, pch=16, col=Col1, cex=1.25)
   
   ## Data 2
   lines(DF$DY, DF$NC2, lwd=2, lty=1, col=Col2)
   points(DF$DY, DF$NC2, pch=15, col=Col2, cex=1.25)
   
   ## Plottind the differences between data points
   ## --------------------------------------------
   points(DF$DY, DF$Dif, type="h", lwd=5, col=Col3)

   ## Customising Axes
   ## ----------------
   ## Customising axes 1
   #abline(h=0)
   if (zone == "PR") {} else lines(c(min(DF$DY), max(DF$DY)), c(0, 0))

   ## Customising axes 2
   axis(2, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))
   mtext(ylab, side=2, line=2.5)
   
   ## Customising axes 4
   axis(4, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))
   
   ## Drawing the legend of the figure
   ## --------------------------------
   wf <- 0.75
   LDY <- unique(DF$DY)
   XP <- LDY[1]
   XT <- LDY[2]
   X1  <- XP - wf*(XT - XP)
   X2  <- XP + wf*(XT - XP)
   Y1 <- Ymax - 1*Yscl
   Y2 <- Ymax - 2*Yscl
   Y3 <- Ymax - 3*Yscl
   Y4 <- Ymax - 4*Yscl
   lines(c(X1, X2), c(Y1, Y1), col=Col1, lty=1, lwd=2)
   points(XP, Y1, pch=16, col=Col1)
   text(XT, Y1, labels=legend[1], col=Col1, adj=0)
   lines(c(X1, X2), c(Y2, Y2), col=Col2, lty=1, lwd=2)
   points(XP, Y2, pch=15, col=Col2)
   text(XT, Y2, labels=legend[2], col=Col2, adj=0)
   lines(c(XP, XP), c(Y3, Y4), col=Col3, lwd=5)
   text(XT, Y3, labels="|diff|", col=Col3, adj=0)   
   

}  ## End plotWFD.Comp

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.Comp function
## ************************************************************************************************
