## ************************************************************************************************
## plotWFD.Comparison function definition
## --------------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics
##  - used for plotting wood formation dynamics for two source of data to compare
##     (e.g., raw vs. stadardised data)
##  - Previous ring measurements are not plotted
##
## Arguments:
##   - data: data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, CZ, EZ, WZ, MZ, CZb, EZb, WZb, MZb
##
## Outputs:
##   - four plots for the four phases of differentiation per trees
##
## Versions:
##   	1.0-0. Defining this new function (10/10/2016)
##   	1.1-1. Improving this function (11-18/10/2016)
##   	1.1-2. Improving this function (6-7/04/2017)
##
## Started: 10 October 2016
## Last modifications: 7 April 2017
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.Comparison <- function(data) {
   
   # message("--> Entering plotWFD.Trend function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Additional data for plotting: month characteristics
   ## ---------------------------------------------------
   MonthId <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
   FirstDay <- c(1, 32, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335)
   MiddleDay <- c(16, 45, 75, 105, 136, 166, 197, 228, 258, 289, 319, 350)
   
   ## Setting the figure Layout for 5 plots
   layout(1:5, heights=c(1.3, 0.98, 0.98, 0.98, 1.3))
   
   ## Setting the figure title
   main <- paste("Site: ", DF$Site[1], " - Year: ", DF$Year[1], " - Tree: ", DF$Tree[1], sep="")
   
   
   ## Plotting cambial cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 4, 2))
   
   ## Calling plotWFD.Comp
   ## --------------------
   CCDF <- DF[, c("Site", "Year", "Tree", "Species", "DY", "CZ.x", "CZ.y")]
   names(CCDF) <- c("Site", "Year", "Tree", "Species", "DY", "NC1", "NC2")
   Ylab.CC <- "Cambial cells"
   plotWFD.Comp(CCDF, zone="CZ", hue=0.30, ylab=Ylab.CC, legend=c("raw", "stan"))

   ## Customising axes 3
   ## ------------------
   axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
   axis(3, at=MiddleDay, tick=TRUE, labels=MonthId)
   
   ## Adding a title to the plot
   ## --------------------------
   mtext(main, side=3, line=2, cex=1.25, font=2)
   
   
   ## Plotting enlarging cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.Comp
   ## --------------------
   ECDF <- DF[, c("Site", "Year", "Tree", "Species", "DY", "EZ.x", "EZ.y")]
   names(ECDF) <- c("Site", "Year", "Tree", "Species", "DY", "NC1", "NC2")
   Ylab.EC <- "Enlarging"
   plotWFD.Comp(ECDF, zone="EZ", hue=0.60, ylab=Ylab.EC, legend=c("raw", "stan"))
   

   ## Plotting thickening cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.Comp
   ## --------------------
   WCDF <- DF[, c("Site", "Year", "Tree", "Species", "DY", "WZ.x", "WZ.y")]
   names(WCDF) <- c("Site", "Year", "Tree", "Species", "DY", "NC1", "NC2")
   Ylab.WC <- "Thickening"
   plotWFD.Comp(WCDF, zone="WZ", hue=0.00, ylab=Ylab.WC, legend=c("raw", "stan"))
   
   
   ## Plotting mature cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.Comp
   ## --------------------
   MCDF <- DF[, c("Site", "Year", "Tree", "Species", "DY", "MZ.x", "MZ.y")]
   names(MCDF) <- c("Site", "Year", "Tree", "Species", "DY", "NC1", "NC2")
   Ylab.MC <- "Mature"
   plotWFD.Comp(MCDF, zone="MZ", hue=0.80, ylab=Ylab.MC, legend=c("raw", "stan"))
   
   
   ## Plotting standardisation ratio
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(4, 4, 0, 2))
   
   ## Calling plotWFD.Comp
   ## --------------------
   PRDF <- DF[, c("Site", "Year", "Tree", "Species", "DY", "PR", "Ratio")]
   names(PRDF) <- c("Site", "Year", "Tree", "Species", "DY", "NC1", "NC2")
   PRDF$NC2 <- PRDF$NC2 * mean(PRDF$NC1)
   Ylab.PR <- "Previous ring"
   plotWFD.Comp(PRDF, zone="PR", hue=0.15, ylab=Ylab.PR, legend=c("raw", "ratio * mean"))
   
   ## Customising x axis of the figure
   ## --------------------------------
   axis(1, at=DF$DY, labels=FALSE)
   text(x=DF$DY, y=(par("usr")[3] - 3), labels=DF$DY, srt=45, pos=1, xpd=TRUE)
   mtext("Day of Year", side=1, line=2.5)
   
   
   ## Adding CaviaR stamp on the figure
   ## =================================
   mtext("CaviaR.2 - plotWoodFormationDynamics function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)

} # End plotWFD.Comparison

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.Comparison function
## ************************************************************************************************
