## ************************************************************************************************
## plotWFD.MvMGroup function definition
## ------------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics
##  - used for plotting wood formation dynamics for data at group level
##  - when previous ring measurements are not available or not wanted
##
## Arguments:
##   - data: data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, CZ, EZ, WZ, MZ)
##
## Outputs:
##   - four plots for the four phases of differentiation by group of trees
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.0-1. Changing the names of the input variables
##   	1.1-1. Improving this function + changing its name (6-7/04/2017)
##
## Started: 23 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.MvMGroup <- function(data) {
   
   # message("--> Entering plotWFD.MvMGroup function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Additional data for plotting: month characteristics
   ## ---------------------------------------------------
   MonthId <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
   FirstDay <- c(1, 32, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335)
   MiddleDay <- c(16, 45, 75, 105, 136, 166, 197, 228, 258, 289, 319, 350)
   
   ## Setting the figure Layout for 4 plots
   layout(1:4, heights=c(1.3, 0.97, 0.97, 1.3))
   
   ## Setting the figure title
   main <- paste("Site: ", DF$Site[1], " - Year: ", DF$Year[1], " - Species: ", DF$Species[1], sep="")
   
   
   ## Plotting cambial cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 4, 2))
   
   ## Calling plotWFD.MvM
   ## ------------------
   CCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "CZ")]
   names(CCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.CC <- "Cambial cells"
   CCDF.outliers <- plotWFD.MvM(CCDF, zone="CZ", hue=0.30, ylab=Ylab.CC)

   
   ## Customising axes 3
   ## ------------------
   axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
   axis(3, at=MiddleDay, tick=TRUE, labels=MonthId)
   
   ## Adding a title to the plot
   ## --------------------------
   mtext(main, side=3, line=2, cex=1.25, font=2)
   
   
   ## Plotting enlarging cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.MvM
   ## ------------------
   ECDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "EZ")]
   names(ECDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.EC <- "Enlarging"
   ECDF.outliers <- plotWFD.MvM(ECDF, zone="EZ", hue=0.60, ylab=Ylab.EC)
   
   
   ## Drawing the first part of the legend of the figure
   ## --------------------------------------------------
   ## Computing Y scale
   ## -----------------
   Ymax <- max(DF$E, na.rm=TRUE)
   Yscl <- (Ymax) / 10
   
   ## Writing the Legend for median and confidence interval
   ## -----------------------------------------------------
   wf <- 0.9
   LDY <- unique(DF$DY)
   XP <- LDY[length(LDY) - 3]
   XT <- LDY[length(LDY) - 2]
   X1  <- XP - wf*(XT - XP)
   X2  <- XP + wf*(XT - XP)
   YCI1 <- Ymax - 1*Yscl
   YM <- Ymax - 2*Yscl
   YCI2 <- Ymax - 3*Yscl
   polygon(c(X1, X2, X2, X1), c(YCI1, YCI1, YCI2, YCI2), col="grey70", border="grey70")
   lines(c(X1, X2), c(YM, YM), col="grey40", lty=3, lwd=3)
   points(XP, YM, pch=21, col="grey40")
   text(XT, YM, labels="median", col = "grey40", adj=0)


   ## Writing the Legend for mean and confidence interval
   ## -----------------------------------------------------
   X1  <- XP - wf*(XT - XP)
   X2  <- XP + wf*(XT - XP)
   YCI1 <- Ymax - 4*Yscl
   YM <- Ymax - 5*Yscl
   YCI2 <- Ymax - 6*Yscl
   polygon(c(X1, X2, X2, X1), c(YCI1, YCI1, YCI2, YCI2), col="grey50", border="grey50")
   lines(c(X1, X2), c(YM, YM), col="grey20", lty=2, lwd=3)
   text(XT, YM, labels="mean", col = "grey20", adj=0)
   text(XT, YCI2, labels="CI", col="grey50", adj=0)
   points(XP, YM, pch=22, col="grey20")

   ## Writing the Legend for problematic values
   ## -----------------------------------------
   points(XP, YM, col = "red", pch = 1, cex=5)
   text(XT, YCI1 + 0.5*Yscl, labels="outlier", col = "red", adj=0)
   
   
   ## Plotting thickening cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.MvM
   ## ------------------
   WCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "WZ")]
   names(WCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.WC <- "Thickening"
   WCDF.outliers <- plotWFD.MvM(WCDF, zone="WZ", hue=0.00, ylab=Ylab.WC)
   
   ## Drawing the second part of the legend
   ## -------------------------------------
   ## Computing Y scale
   ## -----------------
   Ymax <- max(WCDF$NC, na.rm=TRUE)
   Yscl <- (Ymax) / 10
   
   ## Writing the Legend for missing value signs
   ## ------------------------------------------
   XMVP <- rep(LDY[1], times=5)
   XMVT <- rep(LDY[2], times=5)
   YMVP <- c(1, 3, 5, 7, 9) * Yscl
   CMVP <- c("darkgreen", "lightgreen", "yellow", "orange", "red")
   points(XMVP, YMVP, type="p", pch=19, col=CMVP, cex=2)
   text(XMVT, YMVP, labels=c( "0%", "< 25%", "< 50%", "< 75%", "> 75%"), adj=0.3)
   
   ## Writing the title of the second part of the legend
   ## --------------------------------------------------
   text(LDY[1] - 6, 11*Yscl, labels="% of missing values", adj=0)
   
   
   ## Plotting mature cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(4, 4, 0, 2))
   
   ## Calling plotWFD.MvM
   ## ------------------
   MCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "MZ")]
   names(MCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.MC <- "Mature"
   MCDF.outliers <- plotWFD.MvM(MCDF, zone="MZ", hue=0.80, ylab=Ylab.MC)

   ## Drawing the third part of the legend and writting x axis label
   ## ---------------------------------------------------------------
   ## Computing Y scale
   ## -----------------
   Ymax <- max(MCDF$NC, na.rm=TRUE)
   Yscl <- (Ymax) / 10
   
   ## Writing the Legend for the maximal difference between points
   ## ------------------------------------------------------------
   XMD <- rep(LDY[1], times=2)
   YMD <- c(6, 8) * Yscl
   Col.Dif <- hsv(h=0.80, s=0.7, v=0.9, alpha=1) ## Color for difference bars
   lines(XMD, YMD, lwd=5, col=Col.Dif)
   
   ## Writing the title of the second part of the legend
   ## --------------------------------------------------
   text(LDY[1] - 6, 10*Yscl, labels="|mean - median|", adj=0)
   
   ## Customising x axis of the figure
   ## --------------------------------
   abline(h=0)
   axis(1, at=DF$DY, labels=DF$Sample, line=0, cex.axis=0.8)
   mtext("Sample number", side=1, line=2.5)
   

   ## Adding CaviaR stamp on the figure
   ## =================================
   mtext("CaviaR.2 - plotWoodFormationDynamics function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)


   ## Returning a table with detected problematic values
   ## ====================================================================
   ODF <- rbind(CCDF.outliers, ECDF.outliers, WCDF.outliers, MCDF.outliers)
   return(ODF)
   
} # End plotWFD.MvMGroup

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.MvMGroup function
## ************************************************************************************************
