## ************************************************************************************************
## plotWFD.MvM() function definition
## --------------------------------
##
## Purpose: Internal function of plotWFD.Group and plotWFD.Group.PR
##  - used for plotting the number cells aggregated at group level
##
## Arguments:
##   - data: data.frame with imposed column names: Site, Year, Tree, Species, Sample, DY, RF, NC
##   - zone = Name of the inspected zone, for output labelling (default = "NA")
##   - ylab = label of the y axis (default ="Number of cells")
##   - hue = Value of hue (default = 0.6)
##
## Outputs:
##   - one plot of the number of cells / sampling date in the selected differentiation phase
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.1-1. Changing the names of the input variables
##   	1.1-2. Changing the names of scalingYAxis.IUF function
##   	1.1-2. Improving this function + changing its name (6-7/04/2017)
##
## Started: 16 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.MvM <- function(data, zone = "NA", ylab = "Number of cells", hue = 0.6) {
   
   # message("--> Entering plotWFD.MvM function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Formating incoming data
   ## -----------------------
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Tree <- as.factor(DF$Tree)
   DF$Species  <- as.factor(DF$Species)
   DF$Sample <- as.factor(DF$Sample)
   DF$DY <- as.integer(DF$DY)
   DF$NC <- as.integer(DF$NC)
   DF$Zone <- zone
   DF$Dummy <- ifelse(is.na(DF$NC), 0, 1)

   ## Computing median and its confidence interval
   ## --------------------------------------------
   QDF <- unique(DF[, c("Site", "Year", "Species", "Sample", "DY")])
   QDF$Q1 <- tapply(DF$NC, DF$DY, quantile, probs=0.25, na.rm=TRUE) ## First quartile
   QDF$Q2 <- tapply(DF$NC, DF$DY, median, na.rm=TRUE) ## Median
   QDF$Q3 <- tapply(DF$NC, DF$DY, quantile, probs=0.75, na.rm=TRUE) ## Third quartile
   QDF$NV <- tapply(DF$Dummy, DF$DY, sum, na.rm=TRUE) ## Number of values
   QDF$IQR <- QDF$Q3 - QDF$Q1 # Inter Quartile Range
   QDF$SE <-  (1.5 * QDF$IQR) / sqrt(QDF$NV) ## Standard Error
   QDF$LCIL <- QDF$Q2 - QDF$SE # Lower Confindence Interval Limit
   QDF$UCIL <- QDF$Q2 + QDF$SE # Upper Confindence Interval Limit

   ## Computing mean and its confidence interval
   ## ------------------------------------------
   PDF <- unique(DF[, c("Site", "Year", "Species", "Sample", "DY")])
   PDF$Mean <- tapply(DF$NC, DF$DY, mean, na.rm=TRUE) ## Mean
   PDF$SD <- tapply(DF$NC, DF$DY, sd, na.rm=TRUE) ## Sandard Deviation
   PDF$NV <- tapply(DF$Dummy, DF$DY, sum, na.rm=TRUE) ## Number of values
   PDF$SE <-  (qnorm(0.975) * PDF$SD) / sqrt(QDF$NV) ## Standard Error
   PDF$LCIL <- PDF$Mean - PDF$SE # Lower Confindence Interval Limit
   PDF$UCIL <- PDF$Mean + PDF$SE # Upper Confindence Interval Limit
   
   ## Computing the percentage of missing values
   ## ------------------------------------------
   ## Counting every single missing value
   DF$NMV <- ifelse(is.na(DF$NC), 1, 0)
   
   ## Creating a data.frame with the number of missing values per sampling date
   MVDF <- aggregate(DF[, c("NC", "NMV")], # NC is just there to force the correct variable names
                     by=list(Site=DF$Site, Year=DF$Year, Sample=DF$Sample, DY=DF$DY),
                     sum, na.rm=TRUE)
   
   ## Computing the percentage of missing values per dates
   MVDF$PMV <- 100 * (MVDF$NMV / nlevels(DF$Tree))
   
   ## Attributing a color corresponding to the percentage
   ## Function with color code
   attributeColorCode  <- function(Pctg) {
      Color <- "grey90"
      if (Pctg < 101) {Color  <- "red"}
      if (Pctg < 75) {Color  <- "orange"}
      if (Pctg < 50) {Color  <- "yellow"}
      if (Pctg < 25) {Color  <- "lightgreen"}
      if (Pctg < 1) {Color  <- "darkgreen"}
      return (Color)
   } ## End function attributeColorCode
   
   ## Running the color code function 
   MVDF$CMV <- sapply(MVDF$PMV, attributeColorCode)

   ## Computing difference between mean and median for detecting problematic sampling
   ## -------------------------------------------------------------------------------
   DDF <- PDF[, c("DY", "NV")]
   DDF$Dif <- abs(PDF$Mean - QDF$Q2)
   
   ## Looking for problematic sampling dates
   ## --------------------------------------
   ## ==> We will consider as problematic, sampling dates where the mean is outside the confidence interval
   ##     of the median, forcing the confidence interval to be at least ± 1.5 cell around the median.
   
   ## Computing lower boundary
   QDF$LQ <- QDF$Q2 - 1.5
   QDF$LB <- apply(QDF[, c("LCIL", "LQ")], 1, min, na.rm=TRUE)
   QDF$LB <- ifelse(QDF$LB < 0, 0, QDF$LB) ## Because we know we cannot have negative number of cell!
   
   ## Computing upper boundary
   QDF$UQ <- QDF$Q2 + 1.5
   QDF$UB <- apply(QDF[, c("UCIL", "UQ")], 1, max, na.rm=TRUE)
   
   ## Detecting problematic dates
   OLDF <- merge(PDF[, c("Site", "Year", "Species", "Sample", "DY", "Mean")],
                 QDF[, c("DY", "Q2", "LB", "UB")])
   OLDF$LBOL <- ifelse(OLDF$Mean < OLDF$LB, 1, 0)
   OLDF$UBOL <- ifelse(OLDF$Mean > OLDF$UB, 1, 0)
   OLDF$OL <- OLDF$LBOL + OLDF$UBOL
   OLDF$OL <- ifelse(is.na(OLDF$OL) == TRUE, 0, OLDF$OL)
   OLDF$Zone <- zone
   OLDF <- OLDF[OLDF$OL > 0, ]
   ODF <- OLDF[, c("Site", "Year", "Species", "Sample", "DY", "Zone", "Mean", "LB", "UB")]
   names(ODF) <- c("Site", "Year", "Species", "Sample", "DY", "Zone", "NC", "LB", "UB")
   ODF <- ODF[order(ODF$Site, ODF$Year, ODF$Species, ODF$DY), ]


   ## Plotting cells count data
   ## =========================

   ## Setting plot parameters
   ## -----------------------
   Ymax <- max(DF$NC, na.rm=TRUE)
   Scl <- Ymax / 10
   Ymin <- min(DF$NC, na.rm=TRUE)
   CEX <- 2

   ## Defining the plot region
   ## ------------------------
   plot(DF$DY, DF$NC, type="n",
        xlim=c(min(DF$DY), max(DF$DY)), ylim=c(-2*Scl, Ymax + Scl),
        ann=FALSE, axes=FALSE)

   ## defining color code
   ## -------------------
   Col.CIR.med <- hsv(h=hue, s=0.3, v=0.9, alpha=1) ## Color for confidence interval enveloppe 
   Col.CIR.mean <- hsv(h=hue, s=0.5, v=0.9, alpha=1) ## Color for confidence interval enveloppe
   Col.Med <- hsv(h=hue, s=0.7, v=0.5, alpha=1) ## Color for mean
   Col.Mean <- hsv(h=hue, s=0.9, v=0.5, alpha=1) ## Color for median
   Col.Dif <- hsv(h=hue, s=0.7, v=0.9, alpha=1) ## Color for difference bars

   ## Plottind the confidence interval enveloppe for the median
   ## ---------------------------------------------------------
   CIR.med <- data.frame(X=c(QDF$DY, QDF$DY[rev(order(QDF$DY))]), Y=c(QDF$LCIL, QDF$UCIL[rev(order(QDF$DY))]))
   CIR.med$Y <- ifelse(CIR.med$Y < 0, 0, CIR.med$Y)
   CIR.med <- na.omit(CIR.med)
   polygon(CIR.med$X, CIR.med$Y, density=NA, col=Col.CIR.med)

   ## Plottind the confidence interval enveloppe for the mean
   ## ---------------------------------------------------------
   CIR.mean <- data.frame(X=c(PDF$DY, PDF$DY[rev(order(PDF$DY))]), Y=c(PDF$LCIL, PDF$UCIL[rev(order(PDF$DY))]))
   CIR.mean$Y <- ifelse(CIR.mean$Y < 0, 0, CIR.mean$Y)
   CIR.mean <- na.omit(CIR.mean)
   polygon(CIR.mean$X, CIR.mean$Y, density=NA, col=Col.CIR.mean)

   ## Drawing vertical lines for the sampling dates
   ## ---------------------------------------------
   abline(v=DF$DY, lty=1, col="grey90")
   
   ## Plotting the percentage of missing values
   ## -----------------------------------------
   points(MVDF$DY, rep(-Scl, times=length(MVDF$DY)), type="p", pch=19, col=MVDF$CMV, cex=2)

   ## Plottind the maximal differences between data points
   ## ----------------------------------------------------
   DDF <- na.omit(DDF)
   points(DDF$DY, DDF$Dif, type="h", lwd=5, col=Col.Dif)

   ## Plotting the median of the trees
   ## --------------------------------
   MDF <- na.omit(QDF[, c("DY", "Q2")])
   X.Med <- MDF$DY
   Y.Med <- MDF$Q2
   lines(X.Med, Y.Med, lwd=2, lty=3, col=Col.Med)
   points(X.Med, Y.Med, pch=21, col=Col.Med)

   ## Plotting the mean of the trees
   ## ------------------------------
   NDF <- na.omit(PDF[, c("DY", "Mean")])
   X.Mean <- NDF$DY
   Y.Mean <- NDF$Mean
   lines(X.Mean, Y.Mean, lwd=2, lty=2, col=Col.Mean)
   points(X.Mean, Y.Mean, pch=22, col=Col.Mean)

   ## highlighting problematic values
   ## -------------------------------
   X.out <- ODF$DY
   Y.out <- ODF$Mean
   points(X.out, Y.out, type="p", pch=1, lty=3, col="red", cex=5)

   ## Customising Axes
   ## ----------------
   ## Customising axes 1
   abline(h=0)

   ## Customising axes 2
   axis(2, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))
   mtext(ylab, side=2, line=2.5)
   
   ## Customising axes 4
   axis(4, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))


   ## Returning a data frame containing outliers for subsequent uses
   ## ==============================================================
   return(ODF)

}  ## End plotWFD.MvM

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.MvM function
## ************************************************************************************************
