## ************************************************************************************************
## plotWFD.RF function definition
## ------------------------------
##
## Purpose: Internal function of plotWFD.RadialFile and plotWFD.RadialFile.PR
##  - used for plotting wood formation dynamics for raw data in radial file
##
## Arguments:
##   - data: data.frame with imposed column names: Site, Year, Tree, Species, Sample, DY, RF, NC
##   - zone = Name of the inspected zone, for output labelling, default = "NA"
##   - ylab = label of the y axis, default ="Number of cells",
##   - hue = Value of hue, default = 0.6
##
## Outputs:
##   - one plot of wood formation dynamics per tree
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.1-1. Changing the names of the input variables
##   	1.1-2. Changing the names of scalingYAxis.IUF function
##   	1.1-3. Improving this function (6-7/04/2017)
##
## Started: 16 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.RF <- function(data, zone = "NA", ylab = "Number of cells", hue = 0.6) {
   
   # message("--> Entering plotWFD.RF function...")
   DF <- data
   
   ## Preparing cell count data
   ## =========================
   ## Formating incoming data
   ## -----------------------
   DF$Site <- as.factor(DF$Site)
   DF$Year <- as.factor(DF$Year)
   DF$Tree <- as.factor(DF$Tree)
   DF$Species  <- as.factor(DF$Species)
   DF$Sample <- as.factor(DF$Sample)
   DF$DY <- as.integer(DF$DY)
   DF$RF <- as.factor(DF$RF)
   DF$NC <- as.integer(DF$NC)
   DF$Zone <- zone

   ## Computing median, inter quartile range, and upper and lower outer fence
   ## -----------------------------------------------------------------------
   QDF <- unique(DF[, c("Sample", "DY")])
   QDF$Q0 <- tapply(DF$NC, DF$DY, quantile, probs=0, na.rm=TRUE) ## min
   QDF$Q25 <- tapply(DF$NC, DF$DY, quantile, probs=0.25, na.rm=TRUE) ## First quartile
   QDF$Q50 <- tapply(DF$NC, DF$DY, median, na.rm=TRUE) ## Median
   QDF$Q75 <- tapply(DF$NC, DF$DY, quantile, probs=0.75, na.rm=TRUE) ## Last quartile
   QDF$Q100 <- tapply(DF$NC, DF$DY, quantile, probs=1, na.rm=TRUE) ## Last quartile
   QDF$IQR <- QDF$Q75 - QDF$Q25 ## Inter-quartile range (IQR)
   QDF$LF <- QDF$Q25 - 1.5 * QDF$IQR ## Lower outer fence
   QDF$UF <- QDF$Q75 + 1.5 * QDF$IQR ## Upper outer fence
   QDF$Dif <- QDF$Q100 - QDF$Q0 ## difference between max and min
   
   ## Computing median, and median deviation for detecting outliers
   ## -------------------------------------------------------------
   ## !!! We will consider as outliers data points that are outside the range of median ± 50%,
   ## if there is more than 1 cell of difference with the median!!!
   ## We apply this particuliar rule of thumb because most of the time only three RF are recorded,
   ## which makes usual IQR detection innoperant.

   ## Lower boundary
   BDF.Low <- QDF[, c("Sample", "DY", "Q50")]
   BDF.Low$Ratio <- ifelse(is.na(QDF$Q50), NA, QDF$Q50 - 0.75*QDF$Q50)
   BDF.Low$Absol <- QDF$Q50 - 1
   BDF.Low$LB <- ifelse(BDF.Low$Absol < BDF.Low$Ratio, BDF.Low$Absol, BDF.Low$Ratio)
   QDF$LB <- ifelse(BDF.Low$Absol < BDF.Low$Ratio, BDF.Low$Absol, BDF.Low$Ratio)

   ## Upper boundary
   BDF.Up <- QDF[, c("Sample", "DY", "Q50")]
   BDF.Up$Ratio <- ifelse(is.na(QDF$Q50), NA, QDF$Q50 + 0.75*QDF$Q50)
   BDF.Up$Absol <- QDF$Q50 + 1
   BDF.Up$LB <- ifelse(BDF.Up$Absol > BDF.Up$Ratio, BDF.Up$Absol, BDF.Up$Ratio)
   QDF$UB <- ifelse(BDF.Up$Absol > BDF.Up$Ratio, BDF.Up$Absol, BDF.Up$Ratio)
   
   ## Computing the percentage of missing values
   ## ------------------------------------------
   ## Counting every single missing value
   DF$NMV <- ifelse(is.na(DF$NC), 1, 0)
   
   ## Creating a data.frame with the number of missing values per sampling date
   MVDF <- aggregate(DF[, c("NC", "NMV")], # NC is just there to force the correct variable names
                     by=list(Site=DF$Site, Year=DF$Year, Tree=DF$Tree, Sample=DF$Sample, DY=DF$DY),
                     sum, na.rm=TRUE)
   
   ## Computing the percentage of missing values per dates
   MVDF$PMV <- 100 * (MVDF$NMV / nlevels(DF$RF))
   
   ## Defining and running of a function that will attributes a color corresponding
   ## to the percentage of missing values
   ## -----------------------------------------------------------------------------
   attributeColorCode  <- function(Pctg) {
      Color <- "grey90"
      if (Pctg < 101) {Color  <- "red"}
      if (Pctg < 75) {Color  <- "orange"}
      if (Pctg < 50) {Color  <- "yellow"}
      if (Pctg < 25) {Color  <- "lightgreen"}
      if (Pctg < 1) {Color  <- "darkgreen"}
      return (Color)
   } ## End function attributeColorCode
   
   ## Running the color code function 
   MVDF$CMV <- sapply(MVDF$PMV, attributeColorCode)

   ## Looking for outliers
   ## --------------------
   OLDF <- merge(DF[, c("Site", "Year", "Tree", "Sample", "DY", "RF", "NC", "Zone")],
                 QDF[, c("DY", "LB", "UB")])
   OLDF$LBOL <- ifelse(OLDF$NC < OLDF$LB, 1, 0)
   OLDF$UBOL <- ifelse(OLDF$NC > OLDF$UB, 1, 0)
   OLDF$OL <- OLDF$LBOL + OLDF$UBOL
   OLDF$OL <- ifelse(is.na(OLDF$OL) == TRUE, 0, OLDF$OL)
   OLDF <- OLDF[OLDF$OL > 0, ]
   ODF <- OLDF[, c("Site", "Year", "Tree", "Sample", "DY", "RF", "Zone", "NC", "LB", "UB")]
   ODF <- ODF[order(ODF$Site, ODF$Year, ODF$Tree, ODF$DY, ODF$RF), ]


   ## Plotting cell count data
   ## ========================
   ## Setting plot parameters
   ## -----------------------
   Ymax <- max(c(DF$NC, 5), na.rm=TRUE)
   Scl <- Ymax / 10

   ## Defining the plot region
   ## ------------------------
   plot(DF$DY, DF$NC, type="n",
        xlim=c(min(DF$DY), max(DF$DY)), ylim=c(-2*Scl, Ymax + Scl),
        ann=FALSE, axes=FALSE)

   ## defining color code
   ## -------------------
   Col.OFR <- hsv(h=hue, s=0.3, v=1, alpha=1) ## Color for outer fence enveloppe
   Col.IQR <- hsv(h=hue, s=0.5, v=0.9, alpha=1) ## Color for inter-quartile range enveloppe
   Col.Med <- hsv(h=hue, s=0.9, v=0.5, alpha=1) ## Color for median
   Col.Dif <- hsv(h=hue, s=0.7, v=0.9, alpha=1) ## Color for difference bars
   Col.Data.Points <- hsv(h=hue, s=0.9, v=0.5, alpha=1) ## Color for data points

   ## Plottind the upper and lower outer fence enveloppe
   ## --------------------------------------------------
   OFR <- data.frame(X=c(QDF$DY, QDF$DY[rev(order(QDF$DY))]), Y=c(QDF$LF, QDF$UF[rev(order(QDF$DY))]))
   OFR$Y <- ifelse(OFR$Y < 0, 0, OFR$Y)
   OFR <- na.omit(OFR)
   polygon(OFR$X, OFR$Y, density=NA, col=Col.OFR)

   ## Plottind the inter-quartile range enveloppe
   ## -------------------------------------------
   IQR <- data.frame(X=c(QDF$DY, QDF$DY[rev(order(QDF$DY))]), Y=c(QDF$Q25, QDF$Q75[rev(order(QDF$DY))]))
   IQR$Y <- ifelse(IQR$Y < 0, 0, IQR$Y)
   IQR <- na.omit(IQR)
   polygon(IQR$X, IQR$Y, density=NA, col=Col.IQR)

   ## Drawing vertical lines for the sampling dates
   ## ---------------------------------------------
   abline(v=DF$DY, lty=1, col="grey90")
   
   ## Plotting the percentage of missing values
   ## -----------------------------------------
   points(MVDF$DY, rep(-Scl, times=length(MVDF$DY)), type="p", pch=19, col=MVDF$CMV, cex=2)

   ## Plottind the maximal differences between data points
   ## ----------------------------------------------------
   points(QDF$DY, QDF$Dif, type="h", lwd=5, col=Col.Dif)

   ## Plotting the median of the trees
   ## --------------------------------
   MDF <- na.omit(QDF[, c("DY", "Q50")])
   X.Med <- MDF$DY
   Y.Med <- MDF$Q50
   lines(X.Med, Y.Med, lwd=2, lty=1, col=Col.Med)

   ## Plotting data points
   ## --------------------
   ## Defining point type code
   point.type <- rep(c(15:20), times=10)
   i  <- 1

   ## Plotting each radial file successively
   ## --------------------------------------
   for (rf in levels(DF$RF)) { ## Loop for radial file
   
      ## highlighting outliers
      ## ---------------------
      X.out <- ODF$DY[DF$RF == rf]
      Y.out <- ODF$NC[DF$RF == rf]
      points(X.out, Y.out, type="p", pch=1, lty=3, col="red", cex=5)
   
      ## Plotting data points (without missing values)
      ## ---------------------------------------------
      PDF <- na.omit(DF[DF$RF == rf, c("Tree", "Sample", "DY", "NC")])
      X.dtpt <- PDF$DY
      Y.dtpt <- PDF$NC
      points(X.dtpt, Y.dtpt, type="o", pch=point.type[i], lty=3, col=Col.Data.Points, cex=1)
   
      i  <-  i + 1
   } # End loop for Trees

   ## Customising Axes
   ## ----------------
   ## Customising axes 1
   abline(h=0)

   ## Customising axes 2
   axis(2, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))
   mtext(ylab, side=2, line=2.5)

   ## Customising axes 4
   axis(4, at=seq(0, Ymax, by=scalingYAxis.IUF(Ymax)))

   
   ## Returning a data frame containing outliers for subsequent uses
   ## ==============================================================
   return(ODF)

} ## End plotWFD.RF

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.RF function
## ************************************************************************************************
