## ************************************************************************************************
## plotWFD.Tree function definition
## --------------------------------
##
## Purpose: Internal function of plotWoodFormationDynamics
##  - used for plotting wood formation dynamics for data at tree level
##  - when previous ring measurements are not available
##
## Arguments:
##   - data: data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, CZ, EZ, WZ, MZ)
##
## Outputs:
##   - four plots for the four phases of differentiation by group of trees
##
## Versions:
##   	1.0-0. Defining this new function
##   	1.1-3. Improving this function (6-7/04/2017)
##
## Started: 16 May 2014
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWFD.Tree <- function(data) {
   
   # message("--> Entering plotWFD.Tree function...")
   DF <- data
   
   ## Additional data for plotting: month characteristics
   ## ---------------------------------------------------
   MonthId <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
   FirstDay <- c(1, 32, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335)
   MiddleDay <- c(16, 45, 75, 105, 136, 166, 197, 228, 258, 289, 319, 350)
   
   ## Setting the figure Layout for 4 plots
   layout(1:4, heights=c(1.3, 0.97, 0.97, 1.3))
   
   ## Setting the figure title
   main <- paste("Site: ", DF$Site[1], " - Year: ", DF$Year[1], " - Species: ", DF$Species[1], sep="")
   
   
   ## Plotting cambial cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 4, 2))
   
   ## Calling plotWFD.TR
   ## ------------------
   CCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "CZ")]
   names(CCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.CC <- "Cambial cells"
   CCDF.outliers <- plotWFD.TR(CCDF, zone="CZ", hue=0.30, ylab=Ylab.CC)
   
   ## Customising axes 3
   ## ------------------
   axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
   axis(3, at=MiddleDay, tick=TRUE, labels=MonthId)
   
   ## Adding a title to the plot
   ## --------------------------
   mtext(main, side=3, line=2, cex=1.25, font=2)
   
   
   ## Plotting enlarging cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.TR
   ## ------------------
   ECDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "EZ")]
   names(ECDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.EC <- "Enlarging"
   ECDF.outliers <- plotWFD.TR(ECDF, zone="EZ", hue=0.60, ylab=Ylab.EC)
   
   
   ## Drawing the first part of the legend of the figure
   ## --------------------------------------------------
   ## Computing Y scale
   Ymax <- max(c(ECDF$NC, 5), na.rm=TRUE)
   Yscl <- Ymax / 10
   
   # Writing Legend for data points and outliers
   LDY <- unique(DF$DY)
   XP <- LDY[length(LDY) - 3]
   XT <- LDY[length(LDY) - 2]
   points(XP, Ymax - 1*Yscl, col = "grey25", pch = 16, cex=1)
   points(XP, Ymax - 1*Yscl, col = "red", pch = 1, cex=5)
   text(XT, Ymax, labels="obs", col = "grey25", adj=0)
   text(XT + 1, Ymax - 1.5*Yscl, labels="outlier", col="red", adj=0)
   
   ## Writing the Legend for median, interquartile range, and upper and lower fence
   X1  <- XP - 0.5*(XT - XP)
   X2  <- XP + 0.5*(XT - XP)
   YOF1 <- Ymax - 3*Yscl
   YOF2 <- Ymax - 7*Yscl
   YIQ1 <- Ymax - 4*Yscl
   YIQ2 <- Ymax - 6*Yscl
   YM <- Ymax - 5*Yscl
   polygon(c(X1, X2, X2, X1), c(YOF1, YOF1, YOF2, YOF2), col="grey75", border="grey75")
   polygon(c(X1, X2, X2, X1), c(YIQ1, YIQ1, YIQ2, YIQ2), col="grey50", border="grey50")
   lines(c(X1, X2), c(YM, YM), col="grey25", lwd=3)
   text(XT, YM, labels="median", col = "grey25", adj=0)
   text(XT + 1, YM - 1.5*Yscl, labels="IQR", col="grey50", adj=0)
   text(XT + 2, YM + 1.5*Yscl, labels="1.5 IQR", col="grey75", adj=0)
   
   
   ## Plotting thickening cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(0, 4, 0, 2))
   
   ## Calling plotWFD.TR
   ## ------------------
   WCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "WZ")]
   names(WCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.WC <- "Thickening"
   WCDF.outliers <- plotWFD.TR(WCDF, zone="WZ", hue=0.00, ylab=Ylab.WC)
   
   ## Drawing the second part of the legend
   ## -------------------------------------
   ## Computing Y scale
   Ymax <- max(c(WCDF$NC, 5), na.rm=TRUE)
   Yscl <- Ymax / 12
   
   ## Writing the Legend for missing value signs
   XMVP <- rep(LDY[1], times=5)
   XMVT <- rep(LDY[2], times=5)
   YMVP <- Yscl + c(1, 3, 5, 7, 9) * Yscl
   CMVP <- c("darkgreen", "lightgreen", "yellow", "orange", "red")
   points(XMVP, YMVP, type="p", pch=19, col=CMVP, cex=2)
   text(XMVT, YMVP, labels=c( "0%", "< 25%", "< 50%", "< 75%", "> 75%"), adj=0.3)
   
   ## Writing the title of the second part of the legend
   text(LDY[1] - 6, Ymax, labels="% of missing values", adj=0)
   
   
   ## Plotting mature cells
   ## ====================================================================
   ## Setting plot parameters
   ## -----------------------
   par(mar=c(4, 4, 0, 2))
   
   ## Calling plotWFD.TR
   ## ------------------
   MCDF <- DF[, c("Site", "Year", "Tree", "Species", "Sample",  "DY", "MZ")]
   names(MCDF) <- c("Site", "Year", "Tree", "Species", "Sample",  "DY", "NC")
   Ylab.MC <- "Mature"
   MCDF.outliers <- plotWFD.TR(MCDF, zone="MZ", hue=0.80, ylab=Ylab.MC)

   ## Drawing the third part of the legend and writting x axis label
   ## ---------------------------------------------------------------
   ## Computing Y scale
   Ymax <- max(c(MCDF$NC, 5), na.rm=TRUE)
   Yscl <- Ymax / 10
   
   ## Writing the Legend for the maximal difference between points
   XMD <- rep(LDY[1], times=2)
   YMD <- c(5, 8) * Yscl
   Col.Dif <- hsv(h=0.80, s=0.7, v=0.9, alpha=1) ## Color for difference bars
   lines(XMD, YMD, lwd=5, col=Col.Dif)
   
   ## Writing the title of the second part of the legend
   text(LDY[1] - 6, Ymax, labels="max - min", adj=0)
   
   ## Customising x axis of the figure
   ## --------------------------------
   abline(h=0)
   axis(1, at=DF$DY, labels=DF$Sample, line=0, cex.axis=0.8)
   mtext("Sample number", side=1, line=2.5)
   
   
   ## Adding CaviaR stamp on the figure
   ## =================================
   mtext("CaviaR.2 - plotWoodFormationDynamics function",
         side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
   
   
   ## Returning a table with detected outliers
   ## ====================================================================
   ODF <- rbind(CCDF.outliers, ECDF.outliers, WCDF.outliers, MCDF.outliers)
   return(ODF)
   
} ## End plotWFD.Tree

## ------------------------------------------------------------------------------------------------
##                           End plotWFD.Tree function
## ************************************************************************************************
