## *********************************************************************************************
## plotWoodFormationCalendar() function definition
## -----------------------------------------------
##
## Plotting wood formation calendar
## 
## Arguments:
##    - data: data.frame with imposed column names, typically output from computeCriticalDates()
##			function
##		- level (optional): character indicating the plot type to be drawn
##          - type "Tree" --> plots individual tree critical dates and durations
##				- type "Group.Dates" --> plots goup critical dates 
##				- type "Group.Durations" --> plots goup critical durations
##		- main (optional): character containing the plot tittle
##		- sub (optional): character containing theplot subtittle
##		- xlim (optional): numeric vector indicating x-axis min and max values
##
## Output:
##		- a plot of wood formation calendar
##		- a dataframe of the median and median absolute deviation of critical dates for the group
## 
## Versions:
##    1. plot individual critical dates
##		2. plot group critical dates
##		3. plot group phases durations
##		4. Compute median and inter-quartile range
##			4.1. Modifying presentation of mature phase and missing values (15/12/2011)
##			4.2. Changing title name argument for main (01/01/2012)
##    5. General cleaning
##       5.1. Changing variable names for homogeneisation with other CaviaR functions (09/09/2016)
##       5.2. General cleaning of the function (option level = 4 has been removed).
##       5.3-1. Changing option names (11/04/2017)
##
## Started: 7 July 2009
## Last modifications: 11 April 2017
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWoodFormationCalendar <- function(data, level = "Tree", main = " ", sub = " ", xlim = NULL) {
   
   #message("--> Entering plotWoodFormationCalendar function...")
   
   
   ## =============================================================================================   
   ## Declarations and settings
   ## =============================================================================================
   ## Formating variables
   ## -------------------
	DF <- data
	DFl <- is.na(DF)
	nT <- nrow(DF)

	## Additional data for plotting: month characteristics
	## ---------------------------------------------------
	MonthId <- c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
	FirstDay <- c(1, 32, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335)
	MiddleDay <- c(16, 45, 75, 105, 136, 166, 197, 228, 258, 289, 319, 350)
	
	## Replacing NA in the input dataframe by outside-plot-region values
	## -----------------------------------------------------------------
	DFa  <- DF
	DFa$bE <- ifelse(is.na(DF$bE), 0, DF$bE)
	DFa$bW <- ifelse(is.na(DF$bW), 0, DF$bW)
	DFa$bM <- ifelse(is.na(DF$bM), 0, DF$bM)
	DFa$cE <- ifelse(is.na(DF$cE), 400, DF$cE)
	DFa$cW <- ifelse(is.na(DF$cW), 400, DF$cW)
	

	## =============================================================================================  	
	## Plotting individual critical dates if option 1 is selected
	## =============================================================================================  
	if (level == "Tree") {
		
		## Setting the plot region
	   ## =======================
	   ## x-axis range
	   if (is.null(xlim) == TRUE) {
	      xlim <- c(min(DF$bE - DF$bE.ci/2, na.rm=TRUE), max(DF$cW + DF$cW.ci/2, na.rm=TRUE))
	   }
	   
	   ## y-axis range and labels
		ymax <- 25
		ymin <- 1
		YTicks <- c(1, 2, 3, 4, 6, 7, 8, 9, 11, 12, 13, 14, 16, 17, 18, 19, 21, 22, 23, 24)
		YLabs <- rep(c("M", "W", "E", ""), times=5) # Phenophase C temporarily removed

		## Drawing the plot region
		plot(0, 0, xlim=xlim, ylim=c(ymin, ymax), type="n", ann=FALSE, axes=FALSE)


		## Plotting the period for five trees
		## ==================================
		for (i in 1:nT) {
		
			lw <- 3 ## parameter for line width
		
			TreePosition <- 5 * (i - 1)
		
			## Enlarging period
			## ----------------
			y <- TreePosition + 3
			plotPlayMark.IUF(DFa$bE[i], y, DFa$bE.ci[i]/2, direction="left", color="blue")
			plotPlayMark.IUF(DFa$cE[i], y, DFa$cE.ci[i]/2, direction="right", color="blue")

			if ( DFl[i, "bE"] == TRUE | DFl[i, "cE"] == TRUE) {
				X <- seq(DFa$bE[i]+10, DFa$cE[i]-10, 10)
				Y <- rep(y, length(X))
				points(X, Y, pch=3, col="blue")
			}
			else {
				lines(c(DFa$bE[i], DFa$cE[i]), c(y, y), col="blue", lwd=lw)
			}
		
			## Wall thickening period
			## ----------------------
			y <- TreePosition + 2
			plotPlayMark.IUF(DFa$bW[i], y, DFa$bW.ci[i]/2, direction="left", color="red")
			plotPlayMark.IUF(DFa$cW[i], y, DFa$cW.ci[i]/2, direction="right", color="red")
			if ( DFl[i, "bW"] == TRUE | DFl[i, "cW"] == TRUE) {
				X <- seq(DFa$bW[i]+10, DFa$cW[i]-10, 10)
				Y <- rep(y, length(X))
				points(X, Y, pch=3, col="red")
			}
			else {
				lines(c(DFa$bW[i], DFa$cW[i]), c(y, y), col="red", lwd=lw)
			}

			## Mature period
			## -------------
			y <- TreePosition + 1
			if (DFl[i, "bM"] == TRUE) {
				X <- seq(DFa$bM[i]+10, 400, 10)
				Y <- rep(y, length(X))
				points(X, Y, pch=3, col="brown")
			}
			else {	
				plotPlayMark.IUF(DFa$bM[i], y, DFa$bM.ci[i]/2, direction="left", color="brown")	
			}
			
		}

		## Customising axes 1
		## ------------------
		axis(side=1)
		mtext("Day of Year", side=1, line=2.5, cex=1)

		## Customising axes 2
		## ------------------
		axis(side=2, at=YTicks, labels=YLabs, tcl=-0.25, cex.axis=0.5)
		mtext("Phenophase", side=2, line=2, adj=0.5, cex=1)
		
		## Customising axes 3
		## ------------------
		axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
		axis(3, at=MiddleDay, tick=TRUE, tcl=-0.5, labels=MonthId)
		mtext("Tree", side=3, line=0, adj=1.05, cex=1)
		
		## Customising axes 4
		## ------------------
		for (i in 1:5) {
		   mtext(DF$Tree[i], side=4, line=0.5, at=(2.5 + 5*(i-1)), cex=1)
		}

		## Additional formatting
		## ---------------------
		## Drawing separation lines between trees
		abline(h=c(5, 10, 15, 20), lty=3)

		## Drawing separation between months
		for (i in 1:12) abline(v=FirstDay[i], lty=3)

		## Writting additionnal labels on the plot
		## ---------------------------------------
		mtext(paste(main), side=3, line=2.5, adj=0.5, cex=1.25)
		mtext(paste(paste(sub)), side=3, line=2, adj=0, cex=1)
		
		box()
		
		## Adding CaviaR stamp on the figure
		## ---------------------------------
		mtext("CaviaR.2 - plotWoodFormationCalendar function",
		      side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)

	}
	
	## =============================================================================================  
	## Plotting critical dates if option 2 is selected
	## =============================================================================================  
	if (level == "Group.Dates") {
	
		## Setting the plot region
	   ## =======================
	   ## x-axis range
	   if (is.null(xlim) == TRUE) {
	      xlim <- c(min(DF$bE - DF$bE.ci/2, na.rm=TRUE), max(DF$cW + DF$cW.ci/2, na.rm=TRUE))
	   }
	   
	   ## y-axis range and labels
	   ymin <- 0
	   ymax <- 4 
		YLabsCal <- c("M", "W", "E")
		YTicks <- c(1, 2, 3)	

		## Drawing the plot region
		plot(0, 0, xlim=xlim, ylim=c(ymin, ymax), type="n", ann=FALSE, axes=FALSE)

		
		## Plotting the phenophases for the group of trees
		## ===============================================
		## Drawing separation between months
		## ---------------------------------
		for (i in 1:12) {
		   abline(v=FirstDay[i], lty=3)
		}
		
		## Enlarging period
		## ----------------
		y <- ymax - 1
		QF <- quantile(DF$bE, probs=seq(0, 1, 0.25), na.rm = TRUE)
		QL <- quantile(DF$cE, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDateMark.IUF(QF, QL, y, color = "blue")

		## Wall thickening period
		## ----------------------
		y <- y - 1
		QF <- quantile(DF$bW, probs=seq(0, 1, 0.25), na.rm = TRUE)
		QL <- quantile(DF$cW, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDateMark.IUF(QF, QL, y, color = "red")

		## Mature period
		## -------------
		y <- y - 1
		QF <- quantile(DF$bM, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDateMark.IUF(QF, rep(NA, times=5), y, color = "purple")
	
		## Customising axes 1
		## ------------------
		axis(side=1)
		mtext("Day of Year", side=1, line=2.5, cex=1)

		## Customising axes 2
		## ------------------
		axis(side=2, at=YTicks, labels=YLabsCal, cex.axis=0.75)
		mtext("Phenophase", side=2, line=2, adj=0.5, cex=1)
		
		## Customising axes 3
		## ------------------
		axis(3, at=FirstDay, tick=TRUE, tcl=-1, labels=FALSE)
		axis(3, at=MiddleDay, tick=TRUE, tcl=-0.5, labels=MonthId)
	
		## Writting additionnal labels on the plot
		## ---------------------------------------
		mtext(paste(main), side=3, line=2.5, adj=0.5, cex=1.25)
		mtext(paste(paste(sub)), side=3, line=2, adj=0, cex=1)
		mtext(paste("Onset"), side=3, line=-1.5, adj=0.05, cex=1, font=3)
		mtext(paste("Cessation"), side=3, line=-1.5, adj=0.90, cex=1, font=3)

		box()
		
		## Adding CaviaR stamp on the figure
		## ---------------------------------
		mtext("CaviaR.2 - plotWoodFormationCalendar function",
		      side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
		
	}
	
	## =============================================================================================  	
	## Plotting durations if option 3 is selected
	## =============================================================================================  
	
	if (level == "Group.Durations") {
	   
	   ## Setting the plot region
	   ## =======================
	   ## x-axis range
	   if (is.null(xlim) == TRUE) {
	      xlim <- c(min(DF$dE - DF$dE.ci/2, na.rm=TRUE), max(DF$dX + DF$dX.ci/2, na.rm=TRUE))
	   }
	   
	   ## y-axis range and labels
	   ymin <- 0
	   ymax <- 4
		YLabsDur <- c("X", "W", "E")
		YTicks <- c(1, 2, 3)
	
		## Drawing the plot region
		plot(0, 0, xlim=xlim, ylim=c(ymin, ymax), type="n", ann=FALSE, axes=FALSE)

		## Plotting the phenophase durations for the group of trees
		## ========================================================
		## Drawing vertical lines
		## ----------------------
		abline(v=c(0, 50, 100, 150, 200, 250, 300, 350), lty=3)

		## Enlarging period
		## ----------------
		y <- ymax - 1
		Q <- quantile(DF$dE, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDurationBar.IUF(Q, y, color = "blue")

		## Wall thickening period
		## ----------------------
		y <- y - 1
		Q <- quantile(DF$dW, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDurationBar.IUF(Q, y, color = "red")

		## Wood formation period
		## ---------------------
		y <- y - 1
		Q <- quantile(DF$dX, probs=seq(0, 1, 0.25), na.rm = TRUE)
		plotDurationBar.IUF(Q, y, color = "brown")
	
		## Customising axes 1
		## ------------------
		axis(side=1, cex.axis=1)
		mtext("Number of days", side=1, line=2.5, cex=1)

		## Customising axes 2
		## ------------------
		axis(side=2, at=YTicks, labels=YLabsDur, cex.axis=0.75)
		mtext("Phenophase", side=2, line=2, adj=0.5, cex=1)

		## Writting additionnal labels
		## ---------------------------
		mtext(main, side=3, line=2.5, adj=0.5, cex=1.25)
		mtext(sub, side=3, line=1.5, adj=0, cex=1)

		box()
		
		## Adding CaviaR stamp on the figure
		## ---------------------------------
		mtext("CaviaR.2 - plotWoodFormationCalendar function",
		      side=1, line=3, adj=1, font=2, family="HersheySans", cex=0.5)
		
	}
	
	## =============================================================================================  	
	## Additionnal computations
	## ============================================================================================= 
	## Computing the median and the median absolute deviation
	## ------------------------------------------------------
	RDF <- DF[, c("bE", "bW", "bM", "cE", "cW", "dE", "dW", "dX")]
	Median <- apply(RDF, 2, median)
	Q1 <- apply(RDF, 2, quantile, probs=0.25, na.rm = TRUE)
	Q3 <- apply(RDF, 2, quantile, probs=0.75, na.rm = TRUE)
	IQR <- Q3 - Q1
	MAD <- IQR/2
	
	## Writting the results in a list
	## ------------------------------
	ODF <- rbind(Median, Q1)
	ODF <- rbind(ODF, Q3)
	ODF <- rbind(ODF, IQR)
	ODF <- rbind(ODF, MAD)	

	## Return the output list
	return(ODF)

} ## End plotWoodCalendar function

## ------------------------------------------------------------------------------------------------
##                           End plotWoodCalendar function
## ************************************************************************************************