## ************************************************************************************************
## plotWoodFormationDynamics function definition
## ------------------------------------------------------------------------------------------------
##
## Drawing function for plotting intra-annual dynamics of wood formation
## 
## Arguments:
##   - data: a data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, PR
##   - standardised.data: an optional data.frame with imposed column names:
##      Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, PR
##   - level: an optional character string indicating at which level plots must be drawn: 
##      "RF" (default), "Tree", "MvM", "CT" or "Comp"
##   - show.previous.ring: an optional logical indicating if previous-ring measurements must be drawn or not (default)
##   - stat="mean": an optional character string indicating if central tendency must be represented by the "median" 
##    or the "mean"(default).
##
## Output:
##		- plots of wood formation dynamics
##
##
## Versions:
##		4. Draw envelope around the main curve
##			4.1-0. Changing title name argument for main (01/01/2011)
##    5. Works with raw and aggregated data...
##       5.1-0. Works with raw data for cheking consistency between radial files inside a tree
##       5.2-0. Works with aggregated data for cheking trees behaviour inside a group
##       5.3.0. Works with aggregated data for checking group consistency
##       5.4-0. Works with aggregated data for ploting group central tendency
##       5.4-1. Changing of the names of input variables
##       5.4-2. Changing the output for a dataframe
##    6. Adding a new option to compare different types of data at tree level like raw and standised data
##       6.1-0. Comparison of raw and standised data (10/10/2016)
##       6.1-1. Debugging  (17-18/10/2016)
##    7. Refreshing all the options for inner coherency (07/04/2017)
##
## Started: 10 Jully 2009
## Last modifications: 7 April 2016
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

plotWoodFormationDynamics <- function(data, standardised.data,
                                      level = "RF", show.previous.ring = FALSE, stat = "mean") {
   
   # message("--> Entering plotWoodFormationDynamics function...")
   IDF <- data

   ## Selecting which form of the function to use depending on the format of the input dataframe,
   ## and the selected option
   ## ---------------------------------------------------------------------------------------------
   test.na.PR <- na.omit(IDF$PR)
  

   ## =============================================================================================
   ## Radial File Level
   ## If the input dataframe contains raw data (presence of RF column in the dataframe),
   ## and if option "RF" is selected
   ## ---------------------------------------------------------------------------------------------

   ## Problematic case where option level = "RF" but "RF" column is not present in the dataframe
   ## ------------------------------------------------------------------------------------------
   if (level == "RF" && "RF" %in% names(IDF) == FALSE) {
      warning("plotWoodFormationDynamics function execution interrupted...")
      warning("No Radial File (RF) column found!")
   }

   ## Correct case where option level = "RF" and "RF" column is present in the dataframe
   ## ----------------------------------------------------------------------------------
   if (level == "RF" && "RF" %in% names(IDF) == TRUE) {
      
      ## Expanding input data in order to make all missing values explicit with NA
      EDF <- expandCellCountTable(IDF)       
     
      ## If measurements of previous ring are not available (i.e. PR columns contains only NAs)
      ## ... or if show.previous.ring = FALSE
      if (length(test.na.PR) == 0 | show.previous.ring == FALSE) {
         ## Call plotWFD.RadialFile internal function to draw one figure per tree
         ## This figure contains 4 plots (CZ, EZ, WZ, MZ) with 3 curves (RF1, RF2 & RF3) per plot
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Tree=EDF$Tree), plotWFD.RFile)
      } ## End if (length(test.na.PR) == 0 | show.previous.ring == FALSE)
    
      ## If measurements of previous ring measurements are available (i.e. PR columns contains data)
      ## ... and if show.previous.ring = TRUE
      if (length(test.na.PR) > 0 && show.previous.ring == TRUE) {
         ## Call plotWFD.RadialFile.PR internal function to draw one figure per tree
         ## This figure contains 5 plots (CZ, EZ, WZ, MZ, PR) with 3 curves (RF1, RF2 & RF3) per plot 
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Tree=EDF$Tree), plotWFD.RFile.PR)
      } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)     
   } ## End if ("RF" %in% IDF.col.names == TRUE)


   ## =============================================================================================
   ## Tree Level
   ## If the input dataframe contains aggregated data (absence of RF column in the dataframe),
   ##    or if option "Tree" is selected
   ## ---------------------------------------------------------------------------------------------

   ## Case where option level = "Tree" but the data are not yet aggregated by trees
   ## -----------------------------------------------------------------------------
   if (level == "Tree" && "RF" %in% names(IDF) == TRUE) {
      ## Aggregating the input dataframe
      ADF <- aggregateRadialFiles(IDF)
   }

   ## Case where option level = "Tree" and data are ready yet
   ## -------------------------------------------------------
   if (level == "Tree" && "RF" %in% names(ADF) == FALSE) {
      
      ## Expanding input data in order to make all missing values explicit with NA
      EDF <- expandCellCountTable(ADF)
     
      ## If measurements of previous ring are not available (i.e. PR columns contains only NAs),
      ## or if show.previous.ring = FALSE
      if (length(test.na.PR) == 0 | show.previous.ring == FALSE) {
         ## Call plotWFD.Tree internal function to draw one figure per group of trees
         ## This figure contains 4 plots (CZ, EZ, WZ, MZ) with as many curves as trees
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.Tree)
      } ## End if (length(test.na.PR) == 0 | show.previous.ring == FALSE)
  
      ## If measurements of previous ring measurements are available (i.e. PR columns contains data),
      ## and if show.previous.ring = TRUE
      if (length(test.na.PR) > 0 && show.previous.ring == TRUE) {
         ## Call plotWFD.Tree.PR internal function to draw one figure per group of trees
         ## This figure contains 5 plots (CZ, EZ, WZ, MZ, PR) with as many curves as trees
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.Tree.PR)
      } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)    
   } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)


   ## =============================================================================================
   ## Mean versus Median Level
   ## If option "MvM" is selected
   ## ---------------------------------------------------------------------------------------------

   ## Case where option level = "MvM" but the data are not yet aggregated by trees
   ## ------------------------------------------------------------------------------
   if (level == "MvM" && "RF" %in% names(IDF) == TRUE) {
      ## Aggregating the input dataframe
      ADF <- aggregateRadialFiles(IDF)
   }

   ## Case where option level = "MvM" and data are ready yet
   ## -------------------------------------------------------
   if (level == "MvM" && "RF" %in% names(ADF) == FALSE) {
   
      ## Expanding input data in order to make all missing values explicit with NA
      EDF <- expandCellCountTable(ADF)
   
      ## If measurements of previous ring are not available (i.e. PR columns contains only NAs),
      ## or if show.previous.ring = FALSE
      if (length(test.na.PR) == 0 | show.previous.ring == FALSE) {
         ## Call plotWFD.MvMGroup internal function to draw one figure per group of trees
         ## This figure contains 4 plots (C, E, W, M) with 2 curves: one for the median and one for the mean
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.MvMGroup)
      } ## End if (length(test.na.PR) == 0 | show.previous.ring == FALSE)
   
      ## If measurements of previous ring measurements are available (i.e. P columns contains data),
      ## and if show.previous.ring = TRUE
      if (length(test.na.PR) > 0 && show.previous.ring == TRUE) {
         ## Call plotWFD.MvMGroup.PR internal function to draw one figure per group of trees
         ## This figure contains 5 plots (CZ, EZ, WZ, MZ, PR) with as many curves as trees
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.MvMGroup.PR)
      } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)    
   } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)


   ## =============================================================================================
   ## Central Tendency Level
   ## If option "CT" is selected
   ## ---------------------------------------------------------------------------------------------

   ## Case where option level = "CT" but the data are not yet aggregated by trees
   ## ------------------------------------------------------------------------------
   if (level == "CT" && "RF" %in% names(IDF) == TRUE) {
      ## Aggregating the input dataframe
      ADF <- aggregateRadialFiles(IDF)
   }

   ## Case where option level = "CT" and data are ready yet
   ## --------------------------------------------------------
   if (level == "CT" && "RF" %in% names(ADF) == FALSE) {
   
      ## Expanding input data in order to make all missing values explicit with NA
      EDF <- expandCellCountTable(ADF)
   
      ## If measurements of previous ring are not available (i.e. P columns contains only NAs),
      ## or if show.previous.ring = FALSE
      if (length(test.na.PR) == 0 | show.previous.ring == FALSE) {
         ## Call plotWFD.CTendency internal function to draw one figure per group of trees
         ## This figure contains 4 plots (CZ, EZ, WZ, MZ) with 2 curves: one for the median and one for the mean
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.CTendency, stat=stat)
         ## <--^ !!! (see here under for an explanation)
         ## !!! In fact plotWFD.CTendency function does not return any value, so it is a dummy assignation
      } ## End if (length(test.na.PR) == 0 | show.previous.ring == FALSE)
   
      ## If measurements of previous ring measurements are available (i.e. PR columns contains data),
      ## and if show.previous.ring = TRUE
      if (length(test.na.PR) > 0 && show.previous.ring == TRUE) {
         ## Call plotWFD.CTendency.PR internal function to draw one figure per group of trees
         ## This figure contains 5 plots (CZ, EZ, WZ, MZ, PR) with as many curves as trees
         OLST <- by(EDF, list(Site=EDF$Site, Year=EDF$Year, Species=EDF$Species), plotWFD.CTendency.PR, stat=stat)
         ## <--^ !!! (see here under for an explanation)
         ## In fact plotWFD.CTendency function does not return any value.
         ## So these are two dummy assignations made for coherency with...
      } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)    
   } ## End if (length(test.na.PR) > 0 && show.previous.ring == TRUE)
   
   
   ## =============================================================================================
   ## Comparison Level
   ##  If option "Comp" is selected,
   ##  Check that argument standardised.data is provided,
   ##  And check that the input dataframe must contain aggregated data (absence of RF column in the dataframe)
   ## ---------------------------------------------------------------------------------------------
   if (level == "Comp") {
      if (missing(standardised.data)) {
         ## Sending an error message
         stop("When level 'Comp' is selected, argument standardised.data must be provided")
      } else {SDF <- standardised.data}
      
      if ("RF" %in% names(IDF) == TRUE | "RF" %in% names(SDF) == TRUE) {
         ## Sending an error message
         stop("When level 'Comp' is selected, data and standardised.data must be aggregated data frames")
      } else {
         ## Merging the two data frame
         MDF <- merge(IDF[, c("Site", "Year", "Tree", "Species", "DY", "CZ", "EZ", "WZ", "MZ", "PR")],
                      SDF[, c("Site", "Year", "Tree", "Species", "DY", "CZ", "EZ", "WZ", "MZ", "Ratio")],
                      by=c("Site", "Year", "Tree", "Species", "DY"), all=TRUE)
         
         ## Calling plotWFD.Comparison internal function to draw one figure per tree
         ##  This figure contains 5 plots (CZ, EZ, WZ, MZ, PR) with 2 curves (data, standardised data) per plot
         OLST <- by(MDF, list(Site=MDF$Site, Year=MDF$Year, Tree=MDF$Tree), plotWFD.Comparison)
      } ## End else
   } ## End if (level == "Comp")


   ## ====================================================================
   ## Returning a table with detected problematic values (outliers)
   ## -------------------------------------------------------------
   ODF <- data.frame(do.call(rbind, OLST))
   return(ODF)

} ## End plotWoodFormationDynamics function

## ------------------------------------------------------------------------------------------------
##                           End plotWoodFormationDynamics function
## ************************************************************************************************
