###################################################################################################
## plotWoodGrowth function definition
## ==================================
##
## Purpose: visualisation function for plotting Gompertz models with cumulative cell count data
##          for all the trees of a dataset
##
## Arguments:
##   - growth.model.parameters: a list of elements, each containing four data frames:
##    (1) the references of the group (references);
##    (2) the fitted coefficients (coefficients);
##    (3) the values of observations and model predictions (values); and
##    (4) the model simulations for every day of the year.
##		This is typically the output of fitGompertzModels function.
##   - level: an optional character string indicating at which plots must be drawn: 
##    (1) option level ="Tree" (default), represents the three Gompertz curves (+ observations) for all individual trees;
##    (2) option level = "Group", represents all the individual Gompertz curves along with the mean for all the groups of trees;
##    (3) option level = "CT", represents the central tendency for all the groups of trees;
##    (4) option level = "Comp", compare the central tendency for selected groups of trees.
##   - plot.by: a character string indicating which class must be represented on the plot (default = "Species")
##   - hues: a named vector containg color hue (default = NULL)
##   - plot.title: a character string containing the plot title (default = NULL)
##
## Output:
##   - Plots with curves describing modelled wood growth,
##      along with observed cell count data (EWMZ, WMZ, MZ) according to selected option.
##
## Versions:
##   1. Plots of observed cell count data (EWMZ, WMZ, MZ),
##      along with the three fitted Gompertz curves for one tree.
##   2.1-1. Batch processing (11/11/2016)
##   3.1-1. Developping level option: group level (15/11/2016)
##   4.1-1. Developping level option: CT level (21/11/2016)
##   5.1-1. Developping level option: Comp level (29/11/2016)
##   5.1-2. Improving level option: Comp level (13/01/2017)
##   5.1-3. Improvement and homogeneisation (12/04/2017)
##
## Warning:
##   option level = "Comp", only developped for comparing group of species.
##
## Started: 11 June 2012
## Last modifications: 12 April 2017
## Author: Cyrille RATHGEBER - LERFoB UMR1092 - INRA Nancy
##
###################################################################################################

plotWoodGrowth <- function(growth.model.parameters, level = "Tree", plot.by = "Species", hues = NULL, plot.title = NULL) {
   
   #message("--> Entering plotWoodGrowth function...")
   
   ## reading input list
   ## ------------------
   ILST <- growth.model.parameters
   

   ## =============================================================================================
   ## Tree level
   ## If option "Tree" is selected
   ## Calling plotGompertz.Tree function to plot individual Gompertz curves
   ## -------------------------------------------------------------------------------------------------
   if (level == "Tree") {
      Verbose <- lapply(ILST, plotGompertz.Tree)
   } ## End if (level == "Tree")
   
   
   ## =============================================================================================
   ## Group level
   ## If option "Group" is selected
   ## Calling plotGompertz.Group function to plot Gompertz curves by group of trees
   ## ---------------------------------------------------------------------------------------------
   if (level == "Group") {
      GCDF <- getGompertzCoefficients.IUF(ILST)
      CDF <- GCDF[GCDF$Type == "EWMZ",
                  c("Site", "Year", "Species", "Tree", "a", "b", "k", "ti", "tf", "tx")]
      ## Call plotGompertz.Group internal function to draw one plot per group of trees
      ## This plot shows one curve per tree
      Verbose <- by(CDF, list(Site=CDF$Site, Year=CDF$Year, Species=CDF$Species), plotGompertz.Group)
   } ## End if (level == "Group")
   
   
   ## =============================================================================================
   ## Central tendency level
   ## If option "CT" is selected
   ## Calling plotGompertz.CT function to plot mean Gompertz curve by group of trees
   ## ---------------------------------------------------------------------------------------------
   if (level == "CT") {
      WGSDF <- getWoodGrowthSimulations.IUF(ILST)
      SDF <- WGSDF[, c("Site", "Year", "Species", "Tree", "DY", "EWMZ")]
      ## Call plotGompertz.CT internal function to draw one plot per group of trees
      ## This plot shows one Gompertz curve for the mean of the group with its standard error envelope
      Verbose <- by(SDF, list(Site=SDF$Site, Year=SDF$Year, Species=SDF$Species), plotGompertz.CT)
   } ## End if (level == "CT")
   
     
   ## =============================================================================================
   ## Comparison level
   ## If option "Comp" is selected
   ## Calling plotGompertz.Comp function to plot mean Gompertz curves for the groups to compare
   ## !!! For now this function only work for species comparison !!!
   ## ---------------------------------------------------------------------------------------------
   if (level == "Comp") {
      WGSDF <- getWoodGrowthSimulations.IUF(ILST)
      SDF <- WGSDF[, c("Site", "Year", "Species", "Tree", "DY", "EWMZ")]
      ## Call plotGompertz.Comp internal function to draw one plot per group of trees
      ## This plot shows sevral Gompertz curves for the means of the group with their standard error envelopes
      if (plot.by == "Site") {
         Verbose <- by(SDF, list(Year=SDF$Year, Species=SDF$Species), FUN=plotGompertz.Comp, plot.by,
                       hues, plot.title)
      } else if (plot.by == "Year") {
         Verbose <- by(SDF, list(Site=SDF$Site, Species=SDF$Species), FUN=plotGompertz.Comp, plot.by,
                       hues, plot.title)
      } else if (plot.by == "Species") {
         Verbose <- by(SDF, list(Site=SDF$Site, Year=SDF$Year), FUN=plotGompertz.Comp, plot.by,
                       hues, plot.title)
      } else {
         message(paste(plot.by, "is not a correctgrouping variable!"))
         message('please chose between: "Site", "Year", or "Species"')
      }
   } ## End if (level == "Comp")
   
   ## Returning verbose output
   return(Verbose)
      
} ## End function plotWoodGrowth

###################################################################################################
##                           End plotWoodGrowth function
###################################################################################################