## ************************************************************************************************
## standardiseCellCounts function definition
## ------------------------------------------------------------------------------------------------
##
## Purpose -> Data processing function for creating a data frame where values are standardised by the
##            measurements (number of cells or alternatively width) of the previous tree-ring.
## 
## Arguments:
##	   - data: a data frame composed of 13 columns containing the descriptive variables (Site, Year,
##            Species, Tree, Sample, and DY; as well as the quantitative variables (CZ, EZ, WZ, MZ,
##            PR, WMZ, and EWMZ. Typically, this data frame is the output of the aggregateRadialFiles function.
##		- na.rm: a logical indicating if lines where PR is NA should be left as they are without applying any
##             standardisation (FALSE by default) or removed from the record (TRUE).
##
## Outputs:
##		- A dataframe of the same format containing the standardised values. 
##
##
## Versions:
##    1. Development
##    2. General cleaning 
##    2.1-1. Changing variable names (2016/09/27)
##
## Started: 25 January 2011
## Last modifications: 28 September 2016
## Author: Cyrille RATHGEBER - INRA Nancy
##
## ------------------------------------------------------------------------------------------------

standardiseCellCounts <- function(data, na.rm = "FALSE") {
   
   #message("--> Entering standardiseCellCounts function...")
   
   ## =============================================================================================   
   ## Declarations and settings
   ## =============================================================================================

	IDF <- data
	
	## =============================================================================================   
	## Computations
	## =============================================================================================
	## Computing the average value of the previous ring for each tree
	## --------------------------------------------------------------
	IDF$Tree <- as.factor(IDF$Tree)	
	Means <- aggregate(IDF[, c("PR")], by=list(IDF$Tree), FUN=mean, na.rm=TRUE)
	names(Means) <- c("Tree", "PR.mean")

	## Computing the correction coeficient for each sample
	## ---------------------------------------------------
	DF <- merge(IDF, Means)
	
	if (na.rm=="TRUE")  { DF$Ratio <- DF$PR / DF$PR.mean }
	if (na.rm=="FALSE") { DF$Ratio <- ifelse(is.finite(DF$PR / DF$PR.mean), DF$PR / DF$PR.mean, 1) }
	
	DF <- DF[is.finite(DF$Ratio)=="TRUE", ]
	
	## Computing the standardised value for each sample
	## ------------------------------------------------
	ODF <- DF[, c("Site", "Year", "Tree", "Species", "Sample", "DY", "Ratio")]
	ODF$CZ   <- round(DF$CZ   / DF$Ratio, digits=2)
	ODF$EZ   <- round(DF$EZ   / DF$Ratio, digits=2)
	ODF$WZ   <- round(DF$WZ   / DF$Ratio, digits=2)
	ODF$MZ   <- round(DF$MZ   / DF$Ratio, digits=2)
	ODF$WMZ  <- round(DF$WMZ  / DF$Ratio, digits=2)
	ODF$EWMZ <- round(DF$EWMZ / DF$Ratio, digits=2)
	
	## Returning output dataframe
	## --------------------------
	ODF <- ODF[order(ODF$Tree), ]
	
	return(ODF)

} ## End function standardiseCellCounts

## ------------------------------------------------------------------------------------------------
##                           End standardiseCellCounts function
## ************************************************************************************************