\name{computeRingCellNb}

\alias{computeRingCellNb}

\title{computeRingCellNb}

\description{
 Computes characteristic ring cell numbers.
}

\usage{
 computeRingCellNb(data, stat = "median")
}

\arguments{
  \item{data}{a data frame composed of 13 columns containing the descriptive variables (Site, Year,
              Species, Tree, Sample, and DY; as well as the quantitative variables (CZ, EZ, WZ, MZ).
              Typically, this data frame is the output of the \code{\link{aggregateRadialFiles}} or
              \code{\link{standardiseCellCounts}} functions,
              alternativelly it can also work on raw data (see \code{\link{GRA}}).}
  \item{stat}{an optional string of characters indicating if the function must compute the mean
              or the median (default) as summary statistics.}
}

\details{
 This function computes characteristic tree ring cell numbers (median or mean depending on the option
 selected) as well as their associated measure of variation: mean absolute deviation, mean absolute error,
 and corrected mean absolute error for comparison with standard error (MAD, MAE, MAE.cor),
 or standard deviation and error (SD and SE)) for each tree of the data set. \cr
 The characteristic tree ring cell numbers are: the final ring cell number (RCN), the initial number of
 dormant cambial cells (ICN) and the final number of dormant cambial cells (FCN). \cr
 This function computes also the signal to noise ratio (SNR) for RCN.
}

\value{
 A data frame composed of 19 columns containing the computed characteristic tree ring cell numbers (RCN,
 ICN and FCN) mean or median (e.g. RCN.median or RCN.mean), \cr
 along with their associated measures of variations (e.g. RCN.MAD, RCN.MAE, RCN.MAE.cor or RCN.SD,
 RCN.SE), and number of observations (e.g. RCN.CNT), and signal-to-noise ratio (RCN.SNR). \cr
}

\note{
 Version: 3.1-2. \cr
 Development started: 25 March 2010. \cr
 Last modifications: 06 October 2016.
}

\author{
 Cyrille Rathgeber - LERFoB, UMR1092 - INRA Nancy (France) \cr
 Email: \email{cyrille.rathgeber@inra.fr}
}

\references{
 Rathgeber C.B.K., Longuetaud F., Mothe F., Cuny H. & Le Moguedec G. 2011.
 Phenology of wood formation: Data processing, analysis and visualisation using R (Package CAVIAR).
 \emph{Dendrochronologia} \bold{29}: 139-149. \cr
 
 Rathgeber C.B.K., Rossi S.& Bontemps J.-D. 2011.
 Cambial activity related to tree size in a mature silver-fir plantation.
 \emph{Annals of Botany} \bold{108}: 429-438.
}

\seealso{
 \code{
  \link{aggregateRadialFiles},
  \link{computeBootstrapTest},
  \link{computeCriticalDates},
  \link{computeGompertzCellTimings},
  \link{fitGompertzModels},
  \link{expandCellCountTable},
  \link{GRA},
  \link{overviewCellCountTable},
  \link{plotCleavelandDotChart},
  \link{plotWoodFormationCalendar},
  \link{plotWoodFormationDynamics},
  \link{plotWoodGrowth},
  \link{readExcelCountTable},
  \link{standardiseCellCounts}
  }
}

\examples{
 ## Loading the training dataset
 ## ============================
 data(GRA)

 ## Aggregating and standardising raw data
 ## ======================================
 ## Computing the median for Grandfontaine raw data
 GRA.med <- aggregateRadialFiles(data=GRA)

 ## Standising data aggregated using median
 GRA.med.sta <- standardiseCellCounts(GRA.med)

 ## Computing the mean for Grandfontaine raw data:
 GRA.mean <- aggregateRadialFiles(data=GRA, stat="mean")

 ## Standising data aggregated using mean
 GRA.mean.sta <- standardiseCellCounts(GRA.mean)

 ## Computing final total ring cell number using medians
 ## ====================================================
 ## On raw data using the median (default)
 GRA_RCN.med <- computeRingCellNb(GRA)

 ## On data aggregated using median
 GRA.med_RCN.med <- computeRingCellNb(GRA.med)

 ## On standardised data, initially aggregated using median
 GRA.med.sta_RCN.med <- computeRingCellNb(GRA.med.sta)

 ## Creating a csv file containing output data
 write.csv2(GRA.med.sta_RCN.med, file = paste("Median RCN for GRA.med.sta ", Sys.Date(), ".csv", sep=""),
            row.names = FALSE)

 ## Evaluating the standardisation process using means
 ## =====================================================
 ## On raw data using the mean
 GRA_RCN.mean <- computeRingCellNb(GRA, stat="mean")

 ## On data aggregated using mean
 GRA.mean_RCN.mean <- computeRingCellNb(GRA.mean, stat="mean")

 ## On standardised data, initially aggregated using mean
 GRA.mean.sta_RCN.mean <- computeRingCellNb(GRA.mean.sta)

 ## Placing all the SNR computed at different step of the process in the same data frame
 SNR <- data.frame(Site=GRA_RCN.mean$Site, Year=GRA_RCN.mean$Year, Tree=GRA_RCN.mean$Tree, Species=GRA_RCN.mean$Species,
                   Raw=GRA_RCN.mean$RCN.SNR, Mean=GRA.mean_RCN.mean$RCN.SNR, Sta=GRA.mean.sta_RCN.mean$RCN.SNR)

 ## Checking if standardisation is improving the quality of the data tree by tree
 SNR$Dif <- SNR$Sta - SNR$Mean
 SNR$Pos <- ifelse(SNR$Dif > 0, 1, 0)
 SNR$Neg <- ifelse(SNR$Dif < 0, 1, 0)
 SNR$Best <- SNR$Sta * SNR$Pos + SNR$Mean * SNR$Neg

 ## Overall synthesis
 Summary <- apply(SNR[5:11], 2, sum, na.rm=TRUE)
}

\keyword{datagen}
