\name{plotWoodFormationDynamics}
\alias{plotWoodFormationDynamics}

\title{plotWoodFormationDynamics}

\description{
   Plots intra-annual dynamics of wood formation
}

\usage{
   plotWoodFormationDynamics(data, standardised.data, level="RF", show.previous.ring=FALSE, stat="mean")
}

\arguments{
   \item{data}{a data.frame containing cell count data. \cr
      This data.frame must contain raw data, i.e. 12 imposed variables with the following compulsory column names: \cr
      Site, Year, Tree, Species, Sample, DY, RF, CZ, EZ, WZ, MZ, and PR (see \code{\link{GRA}}). \cr
      Alternatively, this data.frames can be outputed from \code{\link{aggregateRadialFiles}} function, \cr
      (in this case the data.frame contains 11 columns: Site, Year, Tree, Species, Sample, DY, RF, C, E, W, M, and P).}
   \item{standardised.data}{an optional data.frame containing standardised cell count data. \cr
      This data.frame must contain aggregated data, i.e. 11 imposed variables with the following compulsory column names: \cr
      Site, Year, Tree, Species, Sample, DY, CZ, EZ, WZ, MZ, and Ratio. \cr
      Typically, this data.frames can be outputed from \code{\link{standardiseCellCounts}} function, \cr
      (in this case the data.frame contains 11 columns: Site, Year, Tree, Species, Sample, DY, RF, C, E, W, M, and P).}
   \item{level}{an optional character string indicating at which level plots must be drawn: \cr
      "RF" (default), "Tree", "MvM", "CT", or "Comp".}
   \item{show.previous.ring}{an optional logical indicating if previous ring measurements must be drawn or not (default).}
   \item{stat}{an optional character string indicating if central tendency must be represented by the "median" \cr
      or the "mean"(default).}
}

\details{
   This function is designed to allow a meticulous checking of cell count data at different levels. \cr
   Basically, it draws a figure containing a plot for each zone of differentiation 
   (+ on for previous ring measurements optionally). \cr
   \enumerate{
      \item level radial file: the figures allow visualisation of all the raw data tree by tree;
      \item level tree: the figures allow visualisation of all the averaged data group of trees by group of trees;
      \item level group: the figures allow the comparison of the means and the medians group of trees by group of trees;
      \item level trend: the figures allow drawing the central tendency (mean or median) group of trees by group of trees;
      \item level comparison: the figures allow visualisation of all raw vs. standardised data tree by tree.}
}

\value{
   An optional data frame of at 8-9 columns containing:
   the unchanged group variables: Site, Year, Species, Tree (when needed), Sample, and DY; \cr
   and the values for problematic dates: NC (Number of cells), LD (Lower boundary), UB (upper boundary). \cr
}

\note{
   Version: 6.1-1. \cr
   Development started: 27 November 2008. \cr
   Last modifications: 18 Octobre 2014.
}

\author{
   Cyrille Rathgeber - LERFoB, UMR1092 - INRA Nancy (France) \cr
   Email: \email{cyrille.rathgeber@inra.fr}
}

\references{
   Rathgeber C.B.K., Longuetaud F., Mothe F., Cuny H. & Le Moguedec G. 2011.
   Phenology of wood formation: Data processing, analysis and visualisation using R (Package CAVIAR).
   \emph{Dendrochronologia} \bold{29}: 139-149. \cr
}

\seealso{
 \code{
  \link{aggregateRadialFiles},
  \link{computeBootstrapTest},
  \link{computeCriticalDates},
  \link{computeGompertzCellTimings},
  \link{computeRingCellNb},
  \link{fitGompertzModels},
  \link{expandCellCountTable},
  \link{GRA},
  \link{overviewCellCountTable},
  \link{plotCleavelandDotChart},
  \link{plotWoodFormationCalendar},
  \link{plotWoodGrowth},
  \link{readExcelCountTable},
  \link{standardiseCellCounts}
  }
}

\examples{
   ## Loading the training dataset
   data(GRA)
   
   ## Checking raw data, looking for outliers
   ## ---------------------------------------
   ## Radial file level without previous ring mesurements
   Rslt.RF <- plotWoodFormationDynamics(GRA, level="RF", show.previous.ring=FALSE)
   
   ## Radial file level with previous ring mesurements
   Rslt.RF.PR <- plotWoodFormationDynamics(GRA, level="RF", show.previous.ring=TRUE)

   ## Checking tree groups, assessing homogeneity
   ## ------------------------------------------
   ## Tree level without previous ring mesurements
   Rslt.TR <- plotWoodFormationDynamics(GRA, level="Tree", show.previous.ring=FALSE)
   
   ## Tree level without previous ring mesurements
   Rslt.TR.PR <- plotWoodFormationDynamics(GRA, level="Tree", show.previous.ring=TRUE)
   
   ## Checking tree groups, assesing consistency
   ## ------------------------------------------
   ## Group level without previous ring mesurements
   Rslt.GR <- plotWoodFormationDynamics(GRA, level="MvM", show.previous.ring=FALSE)
   
   ## Group level with previous ring mesurements
   Rslt.GR.PR <- plotWoodFormationDynamics(GRA, level="MvM", show.previous.ring=TRUE)
   
   ## Presenting tree group trends, showing central tendency
   ## ------------------------------------------------------
   ## Trend level without previous ring mesurements and median
   Rslt.CT.median <- plotWoodFormationDynamics(GRA, level="CT", show.previous.ring=FALSE, stat="median")

   ## Trend level without previous ring mesurements and mean
   Rslt.CT.PR.mean <- plotWoodFormationDynamics(GRA, level="CT", show.previous.ring=FALSE, stat="mean")
   
   ## Trend level with previous ring mesurements and median
   Rslt.CT.median <- plotWoodFormationDynamics(GRA, level="CT", show.previous.ring=TRUE, stat="median")
   
   ## Trend level with previous ring mesurements and mean
   Rslt.CT.PR.mean <- plotWoodFormationDynamics(GRA, level="CT", show.previous.ring=TRUE, stat="mean")
   
   ## Comparing aggregated raw data with stadardised data
   ## ---------------------------------------------------
   ## Computing the mean for Grandfontaine raw data
   GRA.mean <- aggregateRadialFiles(data=GRA, stat="mean")

   ## Standardising averaged data
   GRA.sdd <- standardiseCellCounts(GRA.mean)
   
   ## Comparison between raw and standardised data
   Rslt.CP <- plotWoodFormationDynamics(GRA.mean, GRA.sdd, level="Comp")
}

\keyword{hplot}
